#!/bin/ash

# EXIT CODES
# 0 SUCCESS
# 1 DEVICE COULD NOT BE MOUNTED
# 2 NOT A CONFIGURED OPENSLX DEVICE
# 3 CONFIG FILE IS EMPTY

# prepares openslx partition to be used
# mounts partition if not mounted
# blocks program execution if it is not a openslx partition
# param $1 is the dev to be mounted
# return is done via the mountpath global variable, which is also used here
prepare_usb () {
    local dev=$1
    mountpath=""

    # get mounted device path. if not mounted attempts to do it
    if $(mount | grep -q "/dev/$dev"); then
        echo "$0: Device /dev/$dev is already mounted"
        # get mountpath using the Usage '%' char as delimiter
        # can't use space because of multiple spaces
        # mountpath=$(df | grep -w $dev | cut -d'%' -f2- | cut -d' ' -f2-)
        mountpath=$(mount | sed 's/ type.*//' | grep -w $dev | cut -d' ' -f3- | head -n1)
        # continues execution of users_pendrive
    else
        echo "$0: Device /dev/$dev is not mounted"
        mkdir -p /media/openslx/$dev # creates folder to mount dev on
        if $(mount "/dev/$dev" /media/openslx/$dev); then
            echo "$0: Device /dev/$dev mounted successfully"
            mountpath="/media/openslx/$dev"
            # continues execution of users_pendrive
        else
            echo "$0: Device /dev/$dev could not be mounted"
            exit 1 # stops openslx users_pendrive execution for this dev
        fi
    fi

    # checks if usb has an openslx config
    if [ -e "$mountpath/Openslx/.config" ]; then
        echo "$0: /dev/$dev is an openslx ready drive"
    else
        echo "$0: /dev/$dev is not ready to be used as an openslx drive"
        # umount dev if it was mounted by openslx and doesnt have a config
        if [ "$mountpath" = "/media/openslx/$dev" ]; then
            echo "$0: umounting /dev/$dev"
            umount -f -l "/dev/$dev" # lazy umount to avoid problems
        fi
        exit 2 # stops openslx users_pendrive execution for this dev
    fi
}

dev="$1"

prepare_usb "$dev" #returns at $mountpath
# will only continue if the device that triggered is a valid openslx device
echo "$0: an openslx ready drive is mounted at: $mountpath"

unset IFS
users=$(cat "$mountpath/Openslx/.config");
if [ -z "$users" ]; then
    echo "$0: $dev has an empty config file"
    exit 3 # stops openslx users_pendrive execution for this dev
fi
set $users

if [ ! -e "/home/openslx/usb_users" ]; then
    mkdir -p /home/openslx/usb_users;
fi

# parse config file until EOF
while (( [ $# -gt 0 ] )); do
    # get the respective line for passwd, shadow and group
    # shift advances the line $1 refers to
    # since it doesn't check if there are 3 lines to parse,
    # the program may misbehave in an insecure way
    # if config file is corrupted or invalid.
    passwdf=$(echo $1 | cut -d'=' -f2)
    shift
    shadowf=$(echo $1 | cut -d'=' -f2)
    shift
    groupf=$(echo $1 | cut -d'=' -f2)
    shift

    # parse each line (stored on variables by the code above) for needed info
    username=$( echo $passwdf | cut -d':' -f1 )
    uID=$( echo $passwdf | cut -d':' -f3 )
    localID=$( echo $uID | grep -e "^2....$" -e "^1....$" )

    groupname=$( echo $groupf | cut -d':' -f1 )
    gID=$( echo $groupf | cut -d':' -f3 )
    localGID=$( echo $gID | grep -e "^2....$" -e "^1....$" )

    # local users have to belong to a group with the same name
    if [ "$username" = "$groupname" ]; then
        # checks if the user's id is in the correct range
        # if the user is not local localID and localGID will be empty
        # because grep will not find anything
        if [ -n "$localID" ] && [ -n "$localGID" ]; then
            /opt/openslx/scripts/adduser.sh "$passwdf" "$groupf" "$shadowf"
            #Add to a file what pendrive this user is from
            mount --bind $mountpath/Openslx/$username /home/$username
            echo "/dev/$dev" > "/home/openslx/usb_users/$username"
        fi
    fi
done
