import { AchievementService } from "@/services/achievement.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { achievementSchemas } from "@/db/schema/achievement.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const service = Container.get(AchievementService)

export const achievementRouter = honoWithJwt()
  .post('/create', zValidator('json', achievementSchemas.achievementInputSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const achievement = achievementSchemas.achievementDtoSchema.parse(
          await service.create(input)
        )

        return c.json({achievement})
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create achievement',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/update', zValidator('json', achievementSchemas.achievementUpdateSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const achievement = achievementSchemas.achievementDtoSchema.parse(
          await service.update(input)
        )

        return c.json({achievement})
      } catch (e) {
        console.log(e)
          
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update achievement',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/delete/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const achievement = achievementSchemas.achievementDtoSchema.parse(
        await service.delete(id)
      )

      return c.json({achievement})
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not delete achievement',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })

  export const publicAchievementRouter = new Hono()
    .get('/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const achievement = achievementSchemas.achievementDtoSchema.parse(
          await service.find(id)
        )
        return c.json({achievement})
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not find achievement',
            code: HttpStatus.NOT_FOUND,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.NOT_FOUND
        )
      }
    })
    .get('/', async (c) => {
      try {
        const achievements = achievementSchemas.achievementDtoSchema.array().parse(
          await service.findMany()
        )
        return c.json({achievements})
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not find achievements',
            code: HttpStatus.NOT_FOUND,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.NOT_FOUND
        )
      }
    })