import { ActionService } from "@/services/action.service"
import Container from "typedi"
import { honoWithJwt } from ".."
import { zValidator } from "@hono/zod-validator"
import { actionSchemas } from "@/db/schema/action.schema"
import { createApexError, HttpStatus } from "@/services/error.service"
import { z } from "zod"

const service = Container.get(ActionService)

export const actionRouter = honoWithJwt()
.post('/create', zValidator('json', actionSchemas.input),
  async (c) => {
    try{
      const input = await c.req.valid('json')
      
      const action = actionSchemas.dto.parse(
        await service.create(input)
      )

      return c.json(action)
    } catch (e){
      return c.json(
      createApexError({
        status: 'error',
        message: 'could not create action',
        code: HttpStatus.BAD_REQUEST,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.BAD_REQUEST
    )
    }
})
.get('/actions',
  async (c) => {
    try {
      const actions = z.array(actionSchemas.dto).parse(await service.findMany())

      return c.json({actions})
    } catch (e) {
      return c.notFound()
    }
})
.get('/:name',
  async (c) => {
    try {
      const name = c.req.param('name')
      
      const action = actionSchemas.dto.parse(await service.findByName(name))

      return c.json({ action })
    } catch (e) {
      return c.notFound()
    }
})
.post('/update',
  zValidator('json', actionSchemas.update),
  async (c) => {
    try {
      const input = await c.req.valid('json')
      const action = actionSchemas.dto.parse(await service.update(input))

      return c.json({ action })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not update action',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.delete('/delete/:id',
  async (c) =>{
    try{
      const id: number = +c.req.param('id')
      
      const action = actionSchemas.dto.parse(
        await service.delete(id)
      )

      return c.json(action)
    } catch (e){
      return c.json(
        createApexError({
          status: 'error',
          message: 'could delete action',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
})