import { authSchema } from '@/db/repo/auth.repo'
import type { PasswordRecoveryModel } from '@/db/schema/password-recovery.schema'
import { userStatsSchemas } from '@/db/schema/user-stats.schema'
import { userSchemas } from '@/db/schema/user.schema'
import { AuthService } from '@/services/auth.service'
import { HttpStatus, createApexError } from '@/services/error.service'
import { PasswordRecoveryService } from '@/services/password.service'
import { UserStatsService } from '@/services/user-stats.service'
import { UserService } from '@/services/user.service'
import { zValidator } from '@hono/zod-validator'
import { Hono } from 'hono'
import Container from 'typedi'

const authService = Container.get(AuthService)
const userService = Container.get(UserService)
const userStatsService = Container.get(UserStatsService)
const passwordRecoveryService = Container.get(PasswordRecoveryService)

export const authRouter = new Hono().post(
  '/signin',
  zValidator('json', authSchema),
  async (c) => {
    try {
      const input = await c.req.valid('json')

      const token = await authService.login(input)

      return c.json({ token })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          code: HttpStatus.NOT_FOUND,
          message: 'Invalid or inexistent user',
          path: c.req.routePath,
          suggestion: 'Check your credentials',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  }
)
.post('/signup',
  zValidator('json', userSchemas.userInputSchema),
  async (c) => {
    try {
      const input = await c.req.valid('json')
      
      const userStats = userStatsSchemas.dto.parse(await userStatsService.create())
      input.user_stats_id = userStats.id
      const user = userSchemas.userDtoSchema.parse(
        await userService.create(input)
      )

      return c.json({ user, userStats })
    } catch (e) {
      return c.text('could not create')
    }
})
.post('/request/:email',
  async (c) => {
    try{
      const email: string = c.req.param('email')
      const user = userSchemas.userDtoSchema.parse(
        await userService.findByEmail(email)
      )

      /*
        * When a ticket is generated, the last one is deleted
        */
      const ticketList = await passwordRecoveryService.findByUserId(user.id)
      if(ticketList?.length != 0){
        const lastTicket = ticketList![ticketList!.length - 1]
        if(lastTicket.validToken){
          await passwordRecoveryService.delete(lastTicket.id)
        }
        
      }
      const resetTicket = await passwordRecoveryService.create(user)
      return c.json(resetTicket)

      /**
       * To-do: send email
       * const subject = "Plataforma MEC de Recursos Educacionais - Redefinição de senha"
       * 
       * const content = passwordRecoveryService.emailTemplatePasswordRecovery(resetTicket);
       * await emailService.sendEmail(user.email, subject, content)
       * return c.json({
       *   message: {
       *     status: 'success',
       *     message: 'email sent successfully',
       *     code: HttpStatus.OK
       *   }
       * })
      */
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not send recovery password email',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
})
.post(
  '/reset', 
//url preview: reset?email=admin%40admin.com&token=xEakn2HpEaH8Xvyz8fsntYvmHip8IVP0NZu7bWpjkEY&password=
  async (c) => {
    try {
      /* 
        * Variable declaration section
        */
      const email = c.req.queries('email')
      const password = c.req.queries('password')
      const token = c.req.queries('token')
      if(email == null || password == null|| token == null)
        throw new Error();
      const user = await userService.findByEmail(email[0])
      if(user == null)
        throw new Error();

      const resetTickets = await passwordRecoveryService.findByUserId(user.id)
      if(resetTickets == null)
        throw new Error();

      /*
        * Find reset ticket in database
        */
      let resetTicket: PasswordRecoveryModel | any = null
        resetTickets.forEach(element => {
            if(element.tokenHash == token[0])
                resetTicket = element
        });

      /*
        * Check if the ticket is valid (expired, used, new ticket generated)
        */
      if(passwordRecoveryService.isNotValidTicket(resetTicket))
        throw new Error();
        
      /*
        * Set user new password and update it
        */
      user.password = password[0]
      
      const ret = userSchemas.userDtoSchema.parse(
        await userService.update(user)
      )
              
      await passwordRecoveryService.delete(resetTicket.id)

      return c.json({ user: ret })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not send recovery password email',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)