import { CollectionLikesService } from "@/services/collection-likes.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const associateSchema = z.object({
  collectionId: z.number(),
  userIds: z.array(z.number()),
})

const service = Container.get(CollectionLikesService);

export const collectionLikesRoutes = honoWithJwt()
  .post('/associate', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { collectionId, userIds } = await c.req.valid('json');
        await service.associateCollectionWithLikes(collectionId, userIds);
        return c.json({ message: 'Likes associated successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and user ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  .post('/:collectionId/delete/:userId',
    async (c) => {
      try {
        const collectionId = +c.req.param('collectionId');
        const userId = +c.req.param('userId');
        await service.removeLikesFromCollection(collectionId, [userId]);
        return c.json({ message: 'Likes removed successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to remove likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and user ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/update/likes', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { collectionId, userIds } = await collectionId.req.valid('json');
        await service.updateCollectionLikes(collectionId, userIds);
        return c.json({ message: 'Likes updated successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to update likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and user ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

  export const publicCollectionLikesRoutes = new Hono()
  .get('/:collectionId/likes',
    async (c) => {
      try {
        const collectionId = +c.req.param('collectionId');
        const likes = await service.getLikesByCollection(collectionId);
        return c.json(likes, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection id is valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .get('/:collectionId/likes/:userId/exists', 
    async (c) => {
      try {
        const collectionId  = +c.req.param('collectionId');
        const userId = +c.req.param('userId');
        const exists = await service.isAssociationExists(collectionId, userId);
        return c.json({ exists }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check association',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  .get('/user/:userId/collections',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const collections = await service.getCollectionsByUser(userId);
        return c.json({ collections }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch collections',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
