import { CollectionResourcesService } from "@/services/collection-resources.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const associateSchema = z.object({
    collectionId: z.number(),
    resourceIds: z.array(z.number()),
})

const service = Container.get(CollectionResourcesService);

export const collectionResourcesRoutes = honoWithJwt()
// associate resources with collection
  .post('/associate', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { collectionId, resourceIds } = await c.req.valid('json');
        await service.associateCollectionWithResources(collectionId, resourceIds);
        return c.json({ message: 'Resources associates succesfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate resources',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and resource ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/:collectionId/delete/:resourceId', 
    async (c) => {
      try {
        const collectionId = +c.req.param('collectionId');
        const resourceId = +c.req.param('resourceId');
        await service.removeResourcesFromCollection(collectionId, [resourceId]);
        return c.json({ message: 'Resources removed successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to remove resources',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and resource ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  

export const publicCollectionResourcesRoutes = new Hono()
// get all resources of a collection
.get('/:collectionId/resources',
    async (c) => {
      try {
        const collectionId = +c.req.param('collectionId');
        const resources = await service.getResourcesbyCollection(collectionId);
        return c.json(resources, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch resources',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection id is valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )  
  // check if resource is associated with collection
  .get('/:collectionId/resources/:resourceId/exists',
    async (c) => {
      try {
        const collectionId  = +c.req.param('collectionId');
        const resourceId = +c.req.param('resourceId');
        const exists = await service.isAssociationExists(collectionId, resourceId);
        return c.json({ exists }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check association',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )  
  // get collections of a resource
  .get('/resource/:resourceId/collections',
    async (c) => {
      try {
        const resourceId = +c.req.param('resourceId');
        const collections = await service.getCollectionsbyResource(resourceId);
        return c.json({ collections }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch collections',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )