import { CollectionStatsService } from "@/services/collection-stats.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { collectionStatsSchemas } from "@/db/schema/collection-stats.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const service = Container.get(CollectionStatsService);

export const collectionsStatsRouter = honoWithJwt()
  .post(
    '/create',
    zValidator('json', collectionStatsSchemas.collectionStatsInputSchema),
    async (c) => {
      try {

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.create()
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-views/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateViews(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-downloads/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateDownloads(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-likes/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateLikes(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-shares/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateShares(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-score/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateScore(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update-followers/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.updateFollows(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/delete/:id', async (c) => {
      try {
        const id = +c.req.param('id')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await service.delete(id)
        )

        return c.json({ collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not delete collection stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

export const getCollectionsStats = new Hono()
  .get('/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
        await service.find(id)
      )

      return c.json({ collectionStats })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collection stats',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })
