import { CollectionsService } from "@/services/collections.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { collectionSchemas } from "@/db/schema/collection.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";
import { collectionStatsSchemas } from "@/db/schema/collection-stats.schema";
import { CollectionStatsService } from "@/services/collection-stats.service";

const service = Container.get(CollectionsService);
const serviceStats = Container.get(CollectionStatsService);

export const collectionsRouter = honoWithJwt()
  .post(
    '/create',
    zValidator('json', collectionSchemas.collectionInputSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const collectionStats = collectionStatsSchemas.collectionStatsDtoSchema.parse(
          await serviceStats.create()
        )

        input.collection_stats_id = collectionStats.id
        const collection = collectionSchemas.collectionDtoSchema.parse(
          await service.create(input)
        )

        return c.json({ collection, collectionStats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create collection',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update',
    zValidator('json', collectionSchemas.collectionUpdateSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const collection = collectionSchemas.collectionDtoSchema.parse(
          await service.update(input)
        )

        return c.json({ collection })
      } catch (e) {
        console.log(e)

        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update collection',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/delete/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const collection = collectionSchemas.collectionDtoSchema.parse(
        await service.delete(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not delete collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })
  .post('/delete-permanently/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const collection = collectionSchemas.collectionDtoSchema.parse(
        await service.deletePermanently(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not delete collection permanently',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })
  .post('/restore/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const collection = collectionSchemas.collectionDtoSchema.parse(
        await service.restore(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not restore collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })

export const getCollections = new Hono()
  .get('/all', async (c) => {
    try {
      const collections = collectionSchemas.collectionDtoSchema.array().parse(
        await service.findMany()
      )

      return c.json({ collections })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collections',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })
  .get('/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const collection = collectionSchemas.collectionDtoSchema.parse(
        await service.find(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })




  .get('getActiveCollectionsByUsers/:id_user',
    async (c) => {
      try {
        const id_user = +c.req.param('id_user')
        const collection = collectionSchemas.collectionDtoSchema.array().parse(
          await service.getActiveCollectionsByUsers(id_user)
        )

        return c.json({ collection })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not find collections by user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.NOT_FOUND
        )
      }
    })

  .get('getAllCollectionsByUsers/:id_user', async (c) => {
    try {
      const id = +c.req.param('id_user')

      const collection = collectionSchemas.collectionDtoSchema.array().parse(
        await service.getAllCollectionsByUsers(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })

  .get('getPublicCollectionsByUser/:id_user', async (c) => {
    try {
      const id = +c.req.param('id_user')

      const collection = collectionSchemas.collectionDtoSchema.array().parse(
        await service.getPublicCollectionsByUser(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })

  .get('getPrivateCollectionsByUser/:id_user', async (c) => {
    try {
      const id = +c.req.param('id_user')

      const collection = collectionSchemas.collectionDtoSchema.array().parse(
        await service.getPrivateCollectionsByUser(id)
      )

      return c.json({ collection })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find collection',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })