import { CommentReplyService } from "@/services/comment-reply.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { commentReplySchema } from "@/db/schema/comment-reply.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";
import { UserStatsService } from "@/services/user-stats.service";
import { ResourceStatsService } from "@/services/resource-stats.service";
import { CommentsService } from "@/services/comments.sevice";

const service = Container.get(CommentReplyService)
const serviceUserStats = Container.get(UserStatsService)
const serviceStatsResource = Container.get(ResourceStatsService)
const serviceComment = Container.get(CommentsService)

export const commentReplyRouter = honoWithJwt()

    .post('/create',
        zValidator('json', commentReplySchema.input),
        async (c) => {
            try {
                const input = c.req.valid('json')
                const commentReply = commentReplySchema.dto.parse(await service.create(input))

                await serviceUserStats.updateComments(input.user_id)

                const comment = await serviceComment.find(input.comment_id)
                if (!comment) {
                    throw new Error("Comment not found")
                }
    
                await serviceStatsResource.updateComments(comment.resource_id)

                return c.json(commentReply)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the reply comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


    .post('/update',
        zValidator('json', commentReplySchema.update),
        async (c) => {
            try {
                const input = c.req.valid('json')
                const commentReply = commentReplySchema.dto.parse(await service.update(input))
                return c.json( commentReply )
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the reply comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json update and try again'
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('deleteData/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const comment = commentReplySchema.dto.parse(await service.deleteData(id))

                return c.json(comment)
            }
            catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the reply comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const comment = commentReplySchema.dto.parse(await service.delete(id))

                return c.json(comment)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the reply comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicCommentsReplyRoute = new Hono()

    // get active reply by comment
    .get('/findReplyByComment/:comment_id', async (c) => {
        try {
            const id_comment = +c.req.param('comment_id')
            const comments = commentReplySchema.dto.array().parse(await service.findReplyByComment(id_comment));

            return c.json(comments);
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'Could not find active reply by comment',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'Are you sure the comment exists?',
                }),
                HttpStatus.BAD_REQUEST
            );
        }
    })

    // get all reply by comment
    .get('/findAllReplyByComment/:comment_id', async (c) => {
        try {
            const id_comment = +c.req.param('comment_id')
            const comments = commentReplySchema.dto.array().parse(await service.findAllReplyByComment(id_comment));

            return c.json(comments);
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'Could not find reply by comment',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'Are you sure the comment exists?',
                }),
                HttpStatus.BAD_REQUEST
            );
        }
    })

    // get all reply by user
    .get('/findAllReplyByUser/:id_user', async (c) => {
        try {
            const id_user = +c.req.param('id_user')
            const comments = commentReplySchema.dto.array().parse(await service.findAllReplyByUser(id_user));

            return c.json(comments);
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'Could not find reply by user',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'Are you sure the user exists?',
                }),
                HttpStatus.BAD_REQUEST
            );
        }
    }) 