import { zValidator } from "@hono/zod-validator";
import { honoWithJwt } from "..";
import { commentsSchema } from "@/db/schema/comments.schema";
import Container from "typedi";
import { CommentsService } from "@/services/comments.sevice";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";
import { UserStatsService } from "@/services/user-stats.service";
import { ResourceStatsService } from "@/services/resource-stats.service";

const service = Container.get(CommentsService);
const serviceUserStats = Container.get(UserStatsService)
const serviceStatsResource = Container.get(ResourceStatsService)


export const commentsRouter = honoWithJwt()
    //create a comment
    .post('/create',
        zValidator('json', commentsSchema.input),

        async (c) => {
            try {
                const input = await c.req.valid('json')
                const comment = commentsSchema.dto.parse(await service.create(input))

                await serviceUserStats.updateComments(input.user_id)
                await serviceStatsResource.updateComments(input.resource_id)

                return c.json(comment)
            }
            catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


    .post('/update',
        zValidator('json', commentsSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const comment = commentsSchema.dto.parse(await service.update(input))
                return c.json( comment )
            }
            catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json update and try again'
                    }),
                    HttpStatus.BAD_REQUEST
                )

            }
        }


    )

    .post('deleteData/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const comment = commentsSchema.dto.parse(await service.deleteData(id))

                return c.json(comment)
            }
            catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const comment = commentsSchema.dto.parse(await service.delete(id))

                return c.json(comment)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicCommentsRoute = new Hono()
    .get('/all', async (c) => {
        try {
            const comment = commentsSchema.dto.array().parse(await service.findMany())
            return c.json({ comment })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find any comment',
                    code: HttpStatus.NOT_FOUND,
                    path: c.req.routePath,
                    suggestion: 'are you sure there are comments?',
                }),
                HttpStatus.NOT_FOUND
            )
        }
    }
    )

    // encontrar um comentário com base no id    
    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const comment = commentsSchema.dto.parse(await service.find(id))

                return c.json(comment)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the comment',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure the comment exists?',

                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

     // get all resource by resource
        .get('/findCommentsByResource/:resource_id', async (c) => {
            try {
                const id_resource = +c.req.param('resource_id')
                const comments =commentsSchema.dto.array().parse( await service.findCommentsByResource(id_resource));
    
                return c.json(comments);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Could not find comments by resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Are you sure the resource exists?',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        })

         // get active resource by resource
            .get('/findAllCommentsByResource/:resource_id', async (c) => {
                try {
                    const id_resource = +c.req.param('resource_id')
                    const resource = commentsSchema.dto.array().parse(await service.findAllCommentsByResource(id_resource))
        
                    return c.json(resource)
                } catch (e) {
                    return c.json(
                        createApexError({
                            status: 'error',
                            message: 'could not find the comments',
                            code: HttpStatus.BAD_REQUEST,
                            path: c.req.routePath,
                            suggestion: 'are you sure the resource exists?',
        
                        }),
                        HttpStatus.BAD_REQUEST
                    )
                }
            })



                // get active resource by user
                .get('/findCommentsByUser/:user_id', async (c) => {
                    try {
                        const user_id = +c.req.param('user_id')
                        const resource = commentsSchema.dto.array().parse(await service.findCommentsByUser(user_id))
            
                        return c.json(resource)
                    } catch (e) {
                        return c.json(
                            createApexError({
                                status: 'error',
                                message: 'could not find the comment',
                                code: HttpStatus.BAD_REQUEST,
                                path: c.req.routePath,
                                suggestion: 'are you sure the user exists?',
            
                            }),
                            HttpStatus.BAD_REQUEST
                        )
                    }
                })


                // get all comments by user
                    .get('/findAllCommentsByUser/:user_id', async (c) => {
                        try {
                            const user_id = +c.req.param('user_id')
                            const resources =commentsSchema.dto.array().parse( await service.findAllCommentsByUser(user_id));
                
                            return c.json(resources);
                        } catch (e) {
                            return c.json(
                                createApexError({
                                    status: 'error',
                                    message: 'Could not find the comment',
                                    code: HttpStatus.BAD_REQUEST,
                                    path: c.req.routePath,
                                    suggestion: 'Are you sure the user exists?',
                                }),
                                HttpStatus.BAD_REQUEST
                            );
                        }
                    })