import { ComplaintService } from "@/services/complaints.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { complaintSchemas } from "@/db/schema/complaint.schema";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";

const service = Container.get(ComplaintService);

export const complaintsRouter = honoWithJwt()
.post(
  '/create',
  zValidator('json', complaintSchemas.complaintInputSchema),
  async (c) => {
    try {
      const input = await c.req.valid('json')

      const complaint = complaintSchemas.complaintDtoSchema.parse(
        await service.create(input)
      )

      return c.json({ complaint })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not create complaint',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.post(
  '/update',
  zValidator('json', complaintSchemas.complaintUpdateSchema),
  async (c) => {
    try {
      const input = await c.req.valid('json')

      const complaint = complaintSchemas.complaintDtoSchema.parse(
        await service.update(input)
      )

      return c.json({ complaint })
    } catch (e) {
      console.log(e)

      return c.json(
        createApexError({
          status: 'error',
          message: 'could not update complaint',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.post(
  '/evaluate',
  zValidator('json', complaintSchemas.complaintUpdateSchema),
  async (c) => {
    try {
      const input = await c.req.valid('json')

      const complaint = complaintSchemas.complaintDtoSchema.parse(
        await service.evaluate(input)
      )

      return c.json({ complaint })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not evaluate complaint',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.post('/delete/:id', async (c) => {
  try {
    const id = +c.req.param('id')

    const complaint = complaintSchemas.complaintDtoSchema.parse(
      await service.delete(id)
    )

    return c.json({ complaint })
  } catch (e) {
    return c.json(
      createApexError({
        status: 'error',
        message: 'could not delete complaint',
        code: HttpStatus.BAD_REQUEST,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.BAD_REQUEST
    )
  }
})
.get('/:id', async (c) => {
  try {
    const id = +c.req.param('id')

    const complaint = complaintSchemas.complaintDtoSchema.parse(
      await service.find(id)
    )

    return c.json({ complaint })
  } catch (e) {
    return c.json(
      createApexError({
        status: 'error',
        message: 'could not find complaint',
        code: HttpStatus.NOT_FOUND,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.NOT_FOUND
    )
  }
})
.get('/', async (c) => {
  try {
    const complaints = complaintSchemas.complaintDtoSchema.array().parse(
      await service.findMany()
    )

    return c.json({ complaints })
  } catch (e) {
    return c.json(
      createApexError({
        status: 'error',
        message: 'could not find complaints',
        code: HttpStatus.NOT_FOUND,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.NOT_FOUND
    )
  }
})