import { EducationalStageService } from "@/services/educational-stage.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { educationalStageSchema } from "@/db/schema/educational-stage.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";


const service = Container.get(EducationalStageService);

export const educationalStageRouter = honoWithJwt()

    .post('/create',
        zValidator('json', educationalStageSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const educationalStage = educationalStageSchema.dto.parse(await service.create(input))

                return c.json({ educationalStage })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the educational stage',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', educationalStageSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const educationalStage = educationalStageSchema.dto.parse(await service.update(input))

                return c.json({ educationalStage })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the educational stage',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const educationalStage = educationalStageSchema.dto.parse(await service.delete(id))

                return c.json({ educationalStage })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the educational stage',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


export const publicEducationalStageRouter = new Hono()

    .get('/all',
        async (c) => {
            try {
                const educationalStages = educationalStageSchema.dto.array().parse(await service.findMany())

                return c.json({ educationalStages })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the educational stages',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure there are educational stages?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const educationalStage = educationalStageSchema.dto.parse(await service.find(id))

                return c.json({ educationalStage })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the educational stage',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure there is an educational stage with this id?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )