import { InstitutionService } from "@/services/institution.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { institutionSchemas } from "@/db/schema/institution.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { z } from "zod";

const service = Container.get(InstitutionService)
export const institutionRouter = honoWithJwt()
.post('/create', zValidator('json', institutionSchemas.input),
  async (c) => {
    try{
      const input = await c.req.valid('json')
      const institution = institutionSchemas.dto.parse(
        await service.create(input)
      )
      return c.json(institution)
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not create institution',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })
.get('/institutions',
  async (c) => {
    try{
      const institutions = z.array(institutionSchemas.dto).parse(await service.findMany())
      return c.json(institutions)
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/name/:name',
  async (c) => {
    try{
      const name = c.req.param('name')
      const institution = institutionSchemas.dto.parse(await service.findByName(name))

      return c.json({institution})
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/uf/:uf',
  async (c) => {
    try{
      const uf = c.req.param('uf')
      const institution = institutionSchemas.dto.parse(await service.findByUf(uf))

      return c.json({institution})
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/city/:city',
  async (c) => {
    try{
      const city = c.req.param('city')
      const institution = institutionSchemas.dto.parse(await service.findByCity(city))

      return c.json({institution})
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/cep/:cep',
  async (c) => {
    try{
      const cep = c.req.param('cep')
      const institution = institutionSchemas.dto.parse(await service.findByCep(cep))

      return c.json({institution})
    } catch (e) {
      return c.notFound()
    }
  }
)
.post('/update',
  zValidator('json', institutionSchemas.update),
  async (c) => {
    try{
      const input = await c.req.valid('json')
      const institution = institutionSchemas.dto.parse(await service.update(input))

      return c.json({ institution })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not update institution',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.delete('/delete/:id',
  async (c) => {
    try{
      const id: number = +c.req.param('id')
      const institution = institutionSchemas.dto.parse(
        await service.delete(id)
      )
      return c.json(institution)
    } catch (e) {
      return c.notFound()
    }
  }
)
