import { ItemsService } from "@/services/items.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { itemsSchema } from "@/db/schema/item.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const service = Container.get(ItemsService);

export const itemsRouter = honoWithJwt()

    .post('/create',
        zValidator('json', itemsSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const item = itemsSchema.dto.parse(await service.create(input))

                return c.json({ item })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the item',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', itemsSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const item = itemsSchema.dto.parse(await service.update(input))

                return c.json({ item })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the item',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const item = itemsSchema.dto.parse(await service.delete(id))

                return c.json({ item })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the item',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/active/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const item = itemsSchema.dto.parse(await service.active(id))

                return c.json({ item })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not active the item',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicItemsRouter = new Hono()

.get('/all',
    async (c) => {
        try {
            const items = itemsSchema.dto.array().parse(await service.findMany())

            return c.json({ items })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find the items',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'are you sure there are items?',
                }),
                HttpStatus.BAD_REQUEST
            )
        }
    }
)

.get('/:id',
    async (c) => {
        try {
            const id = +c.req.param('id')
            const item = itemsSchema.dto.parse(await service.findById(id))

            return c.json({ item })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find the item',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'check the id and try again',
                }),
                HttpStatus.BAD_REQUEST
            )
        }
    }
)