import { LanguageService } from "@/services/language.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { languageSchema } from "@/db/schema/language.schema";
import { Hono } from "hono";


const service = Container.get(LanguageService)

export const languageRouter = honoWithJwt()

    .post('/create',
        zValidator('json', languageSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const language = await languageSchema.dto.parse(await service.create(input))

                return c.json({ language })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the language',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', languageSchema.update),
        async (c) => {

            try {
                const input = await c.req.valid('json')
                const language = await languageSchema.dto.parse(await service.update(input))

                return c.json({ language })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the language',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const language = await service.delete(id)

                return c.json({ language })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the language',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicLanguageRouter = new Hono()

    .get('/all',
        async (c) => {
            try {
                const languages = await service.findMany()
                return c.json({ languages })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the languages',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure there are languages in the database?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const language = await languageSchema.dto.parse(await service.find(id))

                return c.json({ language })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the language',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure the language exists?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )