import { LicenseService } from "@/services/license.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { licenseSchema } from "@/db/schema/license.schema";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const service = Container.get(LicenseService);


export const licenseRouter = honoWithJwt()
    .post('/create',
        zValidator('json', licenseSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const license = licenseSchema.dto.parse(await service.create(input))

                return c.json({ license })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the license',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', licenseSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const license = licenseSchema.dto.parse(await service.update(input))

                return c.json({ license })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the license',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/delete/:id',
        async (c) => {
            try {
                const id = + c.req.param('id')
                const license = licenseSchema.dto.parse(await service.delete(id))

                return c.json({ license })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the license',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


export const publicLicenseRouter = new Hono()

    .get('/all', async (c) => {
        try {
            const licenses = licenseSchema.dto.array().parse(await service.findMany())

            return c.json({ licenses })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find the licenses',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'are you sure there are licenses?',
                }),
                HttpStatus.BAD_REQUEST
            )
        }
    }
    )

    .get('/:id', async (c) => {
        try {
            const id = + c.req.param('id')
            const license = licenseSchema.dto.parse(await service.find(id))

            return c.json({ license })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find the license',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'are you sure the resource exists?',
                }),
                HttpStatus.BAD_REQUEST
            )
        }
    }
)