import { NotificationService } from "@/services/notification.service"
import Container from "typedi"
import { honoWithJwt } from ".."
import { zValidator } from "@hono/zod-validator"
import { createApexError, HttpStatus } from "@/services/error.service"
import { z } from "zod"
import { notificationSchemas } from "@/db/relations/notification.relation"

const service = Container.get(NotificationService)

export const notificationRouter = honoWithJwt()
.post('/create', zValidator('json', notificationSchemas.input),
  async (c) => {
    try{
      const input = await c.req.valid('json')
      
      const notification = notificationSchemas.dto.parse(
        await service.create(input)
      )

      return c.json(notification)
    } catch (e){
      return c.json(
      createApexError({
        status: 'error',
        message: 'could not create notification',
        code: HttpStatus.BAD_REQUEST,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.BAD_REQUEST
    )
    }
})
.get('/notifications',
  async (c) => {
    try {
      const notifications = z.array(notificationSchemas.dto).parse(await service.findMany())

      return c.json({notifications})
    } catch (e) {
      return c.notFound()
    }
})
.get('/action/:action_id',
  async (c) =>{
    try {
      const action_id = +c.req.param('action_id')
      
      const notification = notificationSchemas.dto.parse(await service.findByAction(action_id))

      return c.json({ notification })
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/actor-user/:actor_user_id',
  async (c) =>{
    try {
      const actor_user_id = +c.req.param('actor_user_id')
      
      const notification = notificationSchemas.dto.parse(await service.findByActorUser(actor_user_id))

      return c.json({ notification })
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/target-user/:target_user_id',
  async (c) =>{
    try {
      const target_user_id = +c.req.param('target_user_id')
      
      const notification = notificationSchemas.dto.parse(await service.findByTargetUser(target_user_id))

      return c.json({ notification })
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/target-resource/:target_resource_id',
  async (c) =>{
    try {
      const target_resource_id = +c.req.param('target_resource_id')
      
      const notification = notificationSchemas.dto.parse(await service.findByTargetResource(target_resource_id))

      return c.json({ notification })
    } catch (e) {
      return c.notFound()
    }
  }
)
.get('/target-collection/:target_collection_id',
  async (c) =>{
    try {
      const target_collection_id = +c.req.param('target_collection_id')
      
      const notification = notificationSchemas.dto.parse(await service.findByTargetCollection(target_collection_id))

      return c.json({ notification })
    } catch (e) {
      return c.notFound()
    }
  }
)
.post('/update',
  zValidator('json', notificationSchemas.update),
  async (c) => {
    try {
      const input = await c.req.valid('json')
      const notification = notificationSchemas.dto.parse(await service.update(input))

      return c.json({ notification })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not update notification',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.delete('/delete/:id',
  async (c) =>{
    try{
      const id: number = +c.req.param('id')
      
      const notification = notificationSchemas.dto.parse(
        await service.delete(id)
      )

      return c.json(notification)
    } catch (e){
      return c.json(
        createApexError({
          status: 'error',
          message: 'could delete notification',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
})