import { ObjectTypeService } from "@/services/object-type.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { createApexError, HttpStatus } from "@/services/error.service";
import { zValidator } from "@hono/zod-validator";
import { objectTypeSchema } from "@/db/schema/object-type.schema";
import { Hono } from "hono";

const service = Container.get(ObjectTypeService);

export const objectTypeRouter = honoWithJwt()
    .post('/create',
        zValidator('json', objectTypeSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const objectType = objectTypeSchema.dto.parse(await service.create(input))

                return c.json({ objectType })
            } catch (e) {
                return c.json(
                    createApexError(
                    {
                        status: 'error',
                        message: 'could not create the object type',
                        code: 400,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', objectTypeSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const objectType = objectTypeSchema.dto.parse(await service.update(input))

                return c.json({ objectType })
            } catch (e) {
                return c.json(
                    createApexError(
                    {
                        status: 'error',
                        message: 'could not update the object type',
                        code: 400,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const objectType = objectTypeSchema.dto.parse(await service.delete(id))

                return c.json( objectType )
            } catch (e) {
                return c.json(
                    createApexError(
                    {
                        status: 'error',
                        message: 'could not delete the object type',
                        code: 400,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    export const publicObjectTypeRouter = new Hono()

    .get('/all',
        async (c) => {
            try {
                const objectTypes = await service.findMany()
                return c.json(objectTypes)
            } catch (e) {
                return c.json(
                    createApexError(
                    {
                        status: 'error',
                        message: 'could not get the object types',
                        code: 400,
                        path: c.req.routePath,
                        suggestion: 'are you sure there are object types?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const objectType = await service.findById(id)
                return c.json(objectType)
            } catch (e) {
                return c.json(
                    createApexError(
                    {
                        status: 'error',
                        message: 'could not find the object type',
                        code: 400,
                        path: c.req.routePath,
                        suggestion: 'are you sure the object type exists?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

