import { ResourceEducationalStagesService } from "@/services/resource-educational-stages.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { createApexError, HttpStatus } from "@/services/error.service";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";

const associateSchema = z.object({
    resourceId: z.number(),
    educationalStageIds: z.array(z.number()),
});

const service = Container.get(ResourceEducationalStagesService);

export const resourceEducationalStagesRouter = honoWithJwt()

    .post(
        '/associate',
        zValidator('json', associateSchema),
        async (c) => {
            try {
                const { resourceId, educationalStageIds } = await c.req.valid('json');
                await service.associateResourceWithEducationalStages(resourceId, educationalStageIds);
                return c.json({ message: 'Educational stages associated successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to associate educational stages',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .post(
        '/:resourceId/delete/educationalStage/:educationalStageId',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const educationalStageId = +c.req.param('educationalStageId');
                await service.removeEducationalStageFromResource(resourceId, educationalStageId);
                return c.json({ message: 'Educational stage removed successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to remove educational stage',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .post(
        '/update/educationalStages',
        zValidator('json', associateSchema),
        async (c) => {
            try {
                const { resourceId, educationalStageIds } = await c.req.valid('json');
                await service.updateResourceEducationalStages(resourceId, educationalStageIds);
                return c.json({ message: 'Educational stages updated successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to update educational stages',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )


export const publicResourceEducationalStagesRouter = new Hono()

    .get(
        '/:resourceId/educationalStages',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const educationalStages = await service.getEducationalStagesByResource(resourceId);
                return c.json(educationalStages);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to get educational stages',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .get(
        '/:resourceId/educationalStages/:educationalStageId/exists',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const educationalStageId = +c.req.param('educationalStageId');
                const exists = await service.isAssociationExists(resourceId, educationalStageId);
                return c.json({ exists });
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to check association',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .get(
        '/educationalStage/:educationalStageId/resources',
        async (c) => {
            try {
                const educationalStageId = +c.req.param('educationalStageId');
                const resources = await service.getResourcesByEducationalStage(educationalStageId);
                return c.json(resources);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to get resources',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )