import { ResourceLanguageService } from '@/services/resource-language.service';
import { HttpStatus, createApexError } from '@/services/error.service';
import { zValidator } from '@hono/zod-validator';
import { Hono } from 'hono';
import Container from 'typedi';
import { z } from 'zod';
import { honoWithJwt } from '..';

// Definição dos esquemas de validação
const associateSchema = z.object({
  resourceId: z.number(),
  languageIds: z.array(z.number()),
});


const service = Container.get(ResourceLanguageService);

export const resourceLanguageRouter = honoWithJwt()
  // Associa um recurso a várias linguagens
  .post(
    '/associate',
    zValidator('json', associateSchema),
    async (c) => {
      try {
        const { resourceId, languageIds } = await c.req.valid('json');
        await service.associateResourceWithLanguages(resourceId, languageIds);
        return c.json({ message: 'Languages associated successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate languages',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  // Remove a associação entre um recurso e uma linguagem
  .post(
    '/:resourceId/delete/language/:languageId',
    async (c) => {
      try {
        const resourceId = +c.req.param('resourceId');
        const languageId = +c.req.param('languageId');
        await service.removeLanguageFromResource(resourceId, [languageId]);
        return c.json({ message: 'Language removed successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to remove language',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )

    // Atualiza as associações de linguagens de um recurso
    .post(
        '/update/languages',
        zValidator('json', associateSchema),
        async (c) => {
          try {
            const { resourceId, languageIds } = await c.req.valid('json');
            await service.updateResourceLanguages(resourceId, languageIds);
            return c.json({ message: 'Languages updated successfully' }, HttpStatus.OK);
          } catch (e) {
            return c.json(
              createApexError({
                status: 'error',
                message: 'Failed to update languages',
                code: HttpStatus.BAD_REQUEST,
                path: c.req.routePath,
                suggestion: 'Check the input and try again',
              }),
              HttpStatus.BAD_REQUEST
            );
          }
        }
      );

  export const publicResourceLanguageRouter = new Hono()
  // Obtém todas as linguagens associadas a um recurso
  .get(
    '/:resourceId/languages',
    async (c) => {
      try {
        const  resourceId  = +c.req.param('resourceId');
        const languages = await service.getLanguagesByResource(resourceId);
        return c.json(languages);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch languages',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  // Verifica se a associação entre um recurso e uma linguagem existe
  .get(
    '/:resourceId/language/:languageId/exists',
    async (c) => {
      try {
        const resourceId  = +c.req.param('resourceId');
        const languageId = +c.req.param('languageId');
        const exists = await service.isAssociationExists(resourceId, languageId);
        return c.json({ exists }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check association',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  // Obtém todos os recursos associados a uma linguagem
  .get(
    '/language/:languageId/resources',
    async (c) => {
      try {
        const languageId = +c.req.param('languageId');
        const resources = await service.getResourcesByLanguage(languageId);
        return c.json({ resources }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch resources',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
