import { ResourceLikesService } from "@/services/resource-likes.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const associateSchema = z.object({
  resourceId: z.number(),
  userIds: z.array(z.number()),
})

const service = Container.get(ResourceLikesService);

export const resourceLikesRoutes = honoWithJwt()
// the user has liked the resource 
  .post('/associate', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { resourceId, userIds } = await c.req.valid('json');
        await service.associateResourceWithLikes(resourceId, userIds);
        return c.json({ message: 'Likes associated successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and user ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  // remove likes from resource, the user has unliked the resource
  .post('/:resourceId/delete/:userId',
    async (c) => {
      try {
        const resourceId = +c.req.param('resourceId');
        const userId = +c.req.param('userId');
        await service.removeLikesFromResource(resourceId, [userId]);
        return c.json({ message: 'Likes removed successfully' }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to remove likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the collection and user ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  
export const publicResourceLikesRoutes = new Hono()
// get likes by resource id
  .get('/:resourceId',
    async (c) => {
      try {
        const resourceId = +c.req.param('resourceId');
        const likes = await service.getLikesByResource(resourceId);
        return c.json(likes, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch likes',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the resource id is valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

  // verify if association exists, the user has liked the resource
  .get('/:resourceId/likes/:userId/exists', 
    async (c) => {
      try {
        const resourceId  = +c.req.param('resourceId');
        const userId = +c.req.param('userId');
        const exists = await service.isAssociationExists(resourceId, userId);
        return c.json({ exists }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check association',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )

  // get resources liked by user
  .get('/user/:userId/resources', 
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const resources = await service.getResourcesByUser(userId);
        return c.json({ resources }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch resources',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )