import { ResourceStatsService } from "@/services/resource-stats.service";
import { zValidator } from "@hono/zod-validator";
import Container from "typedi";
import { honoWithJwt } from '..';
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";
import { resourceStatsSchema } from "@/db/schema/resource-stats.schema";


const service = Container.get(ResourceStatsService)

export const resourceStatsRouter = honoWithJwt()
    // rota para criar as stats de um recurso
    .post('/create',
        zValidator('json', resourceStatsSchema.input),
        async (c) => {
            try {
                
                const statsResource = resourceStatsSchema.dto.parse(
                    await service.create()
                )

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

// rota para atualizar as stats de um recurso
    .post('/update',
        zValidator('json', resourceStatsSchema.update),
        async (c) => {

            try {
                const input = await c.req.valid('json')
                const statsResource = resourceStatsSchema.dto.parse(await service.update(input))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

// rota para deletar as stats de um recurso
    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const statsResource = resourceStatsSchema.dto.parse(await service.delete(id))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicResourceStatsRouter = new Hono()
//rota para atualizar as views de um recurso, cada chamada soma 1 as views
    .post('/viewUpdate/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const statsResource = resourceStatsSchema.dto.parse(await service.viewUpdate(id))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id resource input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

//rota para atualizar as shares de um recurso, cada chamada soma 1 as shares
    .post('/sharesUpdate/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const statsResource = resourceStatsSchema.dto.parse(await service.sharesUpdate(id))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id resource input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

//rota para atualizar os downloads de um recurso, cada chamada soma 1 aos downloads
    .post('/downloadUpdate/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const statsResource = resourceStatsSchema.dto.parse(await service.downloadUpdate(id))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id resource input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

//rota para pegar todas as stats de todos os recursos    
    .get('/all', async (c) => {
        try {
            const resourceStats = resourceStatsSchema.dto.array().parse(await service.findMany())

            return c.json({ resourceStats })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not get the stats resources',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'are you sure there are stats resources?',
                }),
                HttpStatus.BAD_REQUEST
            )
        }
    }
    )

//rota para pegar as stats de um recurso especifico    
    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const statsResource = resourceStatsSchema.dto.parse(await service.findById(id))

                return c.json({ statsResource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not get the stats resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )
