import { ResourceSubjectsService } from "@/services/resource-subjects.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";


const associateSchema = z.object({
    resourceId: z.number(),
    subjectIds: z.array(z.number()),
});

const service = Container.get(ResourceSubjectsService);

export const resourceSubjectsRouter = honoWithJwt()

    .post(
        '/associate',
        zValidator('json', associateSchema),
        async (c) => {
            try {
                const { resourceId, subjectIds } = await c.req.valid('json');
                await service.associateResourceWithSubjects(resourceId, subjectIds);
                return c.json({ message: 'Subjects associated successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to associate subjects',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .post(
        '/:resourceId/delete/subject/:subjectId',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const subjectId = +c.req.param('subjectId');
                await service.removeSubjectFromResource(resourceId, [subjectId]);
                return c.json({ message: 'Subject removed successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to remove subject',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )


    .post(
        '/update/subjects',
        zValidator('json', associateSchema),
        async (c) => {
            try {
                const { resourceId, subjectIds } = await c.req.valid('json');
                await service.updateResourceSubjects(resourceId, subjectIds);
                return c.json({ message: 'Subjects updated successfully' }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to update subjects',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

export const publicResourceSubjectsRouter = new Hono()

    //obtem todas as disciplinas associadas a um recurso
    .get(
        '/:resourceId/subjects',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const subjects = await service.getSubjectsByResource(resourceId);
                return c.json(subjects);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to get subjects',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )


    .get(
        '/:resourceId/subject/:subjectId/exists',
        async (c) => {
            try {
                const resourceId = +c.req.param('resourceId');
                const subjectId = +c.req.param('subjectId');
                const exists = await service.isAssociationExists(resourceId, subjectId);
                return c.json({ exists }, HttpStatus.OK);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to check association',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )

    .get(
        '/subject/:subjectId/resources',
        async (c) => {
            try {
                const subjectId = +c.req.param('subjectId');
                const resources = await service.getResourcesBySubject(subjectId);
                return c.json(resources);
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'Failed to get resources',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'Check the input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                );
            }
        }
    )
