import { resourceSchema } from "@/db/schema/resource.schema";
import { ResourceService } from "@/services/resource.service";
import { zValidator } from "@hono/zod-validator";
import { honoWithJwt } from '..';
import Container from "typedi";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";
import { resourceStatsSchema } from "@/db/schema/resource-stats.schema";
import { ResourceStatsService } from "@/services/resource-stats.service";

const service = Container.get(ResourceService);
const serviceStats = Container.get(ResourceStatsService);


//aqui teremos a rota de acessar e uma de baixar recurso quando o s3 estiver okay

export const resourceRouter = honoWithJwt()
    // create a resource
    .post('/create',
        zValidator('json', resourceSchema.input),

        async (c) => {
            try {
                const input = await c.req.valid('json')
              
                //cria o stats do recurso correspondente
                const stats = resourceStatsSchema.dto.parse(
                    await serviceStats.create()
                )

                input.resource_stats_id = stats.id
                const resource = resourceSchema.dto.parse(
                    await service.create(input)
                )

                return c.json({ resource, stats })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }

        }
    )

    // update a resource    
    .post('/update',
        zValidator('json', resourceSchema.update),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const resource = resourceSchema.dto.parse(await service.update(input))

                return c.json({ resource })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        })

    // rota para deletar um recurso 
    .post('/deleteData/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const resource = resourceSchema.dto.parse(await service.deleteData(id))

                //onDelete: "cascade" -> presente no stats-resources.schema.ts ja exclui o stats 

                return c.json(resource)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    //rota para dizer que o recurso foi deletado
    .post('/delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const resource = resourceSchema.dto.parse(await service.delete(id))

                return c.json(resource)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    //rota para tornar o recurso ativo
    .post('/active/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const resource = resourceSchema.dto.parse(await service.active(id))

                return c.json(resource)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not active the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


export const publicResourceRouter = new Hono()
    // get all resources
    .get('/all', async (c) => {
        try {
            const resource = resourceSchema.dto.array().parse(await service.findMany())
            return c.json({ resource })
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find any resource',
                    code: HttpStatus.NOT_FOUND,
                    path: c.req.routePath,
                    suggestion: 'are you sure there are resource?',
                }),
                HttpStatus.NOT_FOUND
            )
        }
    }
    )

    // get all resource by user
    .get('/allResourceByUser/:user_id', async (c) => {
        try {
            const user_id = +c.req.param('user_id')
            const resources =resourceSchema.dto.array().parse( await service.allResourceByUser(user_id));

            return c.json(resources);
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'Could not find the resource',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'Are you sure the user exists?',
                }),
                HttpStatus.BAD_REQUEST
            );
        }
    })


    // get active resource by user
    .get('/activeResourceByUser/:user_id', async (c) => {
        try {
            const user_id = +c.req.param('user_id')
            const resource = resourceSchema.dto.array().parse(await service.activeResourceByUser(user_id))

            return c.json(resource)
        } catch (e) {
            return c.json(
                createApexError({
                    status: 'error',
                    message: 'could not find the resource',
                    code: HttpStatus.BAD_REQUEST,
                    path: c.req.routePath,
                    suggestion: 'are you sure the user exists?',

                }),
                HttpStatus.BAD_REQUEST
            )
        }
    })

    // get resource by id
    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const resource = resourceSchema.dto.parse(await service.findById(id))

                return c.json(resource)
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the resource',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure the resource exists?',

                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )
