import { RoleService } from "@/services/role.service"
import Container from "typedi"
import { honoWithJwt } from ".."
import { zValidator } from "@hono/zod-validator"
import { roleSchemas } from "@/db/schema/role.schema"
import { createApexError, HttpStatus } from "@/services/error.service"
import { z } from "zod"

const service = Container.get(RoleService)

export const roleRouter = honoWithJwt()
.post('/create', zValidator('json', roleSchemas.input),
  async (c) => {
    try{
      const input = await c.req.valid('json')
      
      const role = roleSchemas.dto.parse(
        await service.create(input)
      )

      return c.json(role)
    } catch (e){
      return c.json(
      createApexError({
        status: 'error',
        message: 'could not create role',
        code: HttpStatus.BAD_REQUEST,
        path: c.req.routePath,
        suggestion: 'check the input and try again',
      }),
      HttpStatus.BAD_REQUEST
    )
    }
})
.get('/roles',
  async (c) => {
    try {
      const roles = z.array(roleSchemas.dto).parse(await service.findMany())

      return c.json({roles})
    } catch (e) {
      return c.notFound()
    }
})
.get('/:name',
  async (c) => {
    try {
      const name = c.req.param('name')
      
      const role = roleSchemas.dto.parse(await service.findByName(name))

      return c.json({ role })
    } catch (e) {
      return c.notFound()
    }
})
.post('/update',
  zValidator('json', roleSchemas.update),
  async (c) => {
    try {
      const input = await c.req.valid('json')
      const role = roleSchemas.dto.parse(await service.update(input))

      return c.json({ role })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not update role',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  }
)
.delete('/delete/:id',
  async (c) =>{
    try{
      const id: number = +c.req.param('id')
      
      const role = roleSchemas.dto.parse(
        await service.delete(id)
      )

      return c.json(role)
    } catch (e){
      return c.json(
        createApexError({
          status: 'error',
          message: 'could delete role',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
})