import { SubjectsService } from "@/services/subjects.service";
import { createApexError, HttpStatus } from "@/services/error.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { subjectSchema } from "@/db/schema/subject.schema";
import { Hono } from "hono";
import { zValidator } from "@hono/zod-validator";


const service = Container.get(SubjectsService)

export const subjectsRouter = honoWithJwt()
    //rota com token

    .post('/create',
        zValidator('json', subjectSchema.input),
        async (c) => {
            try {
                const input = await c.req.valid('json')
                const subject = await subjectSchema.dto.parse(await service.create(input))

                return c.json({ subject })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not create the subject',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('/update',
        zValidator('json', subjectSchema.update),
        async (c) => {

            try {
                const input = await c.req.valid('json')
                const subject = await subjectSchema.dto.parse(await service.update(input))

                return c.json({ subject })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not update the subject',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the json input and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .post('delete/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const subject = await service.delete(id)

                return c.json({ subject })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not delete the subject',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id  and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

export const publicSubjectsRouter = new Hono()

    .get('/all',
        async (c) => {
            try {
                const subjects = await service.findMany()
                return c.json({ subjects })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the subjects',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'are you sure there are subjects?',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )

    .get('/:id',
        async (c) => {
            try {
                const id = +c.req.param('id')
                const subject = await service.find(id)
                return c.json({ subject })
            } catch (e) {
                return c.json(
                    createApexError({
                        status: 'error',
                        message: 'could not find the subject',
                        code: HttpStatus.BAD_REQUEST,
                        path: c.req.routePath,
                        suggestion: 'check the id and try again',
                    }),
                    HttpStatus.BAD_REQUEST
                )
            }
        }
    )


