import { SubmissionService } from "@/services/submission.service";
import Container from "typedi";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { submissionSchemas } from "@/db/schema/submission.schema";
import { createApexError, HttpStatus } from "@/services/error.service";

const service = Container.get(SubmissionService);

export const submissionRouter = honoWithJwt()
  .post(
    '/create',
    zValidator('json', submissionSchemas.submissionInputSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const submission = submissionSchemas.submissionDtoSchema.parse(
          await service.create(input)
        )

        return c.json({ submission })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create submission',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/update',
    zValidator('json', submissionSchemas.submissionUpdateSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const submission = submissionSchemas.submissionDtoSchema.parse(
          await service.update(input)
        )

        return c.json({ submission })
      } catch (e) {
        console.log(e)

        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update submission',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post(
    '/answer',
    zValidator('json', submissionSchemas.submissionUpdateSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const submission = submissionSchemas.submissionDtoSchema.parse(
          await service.answer(input)
        )

        return c.json({ submission })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not answer submission',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/delete/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const submission = submissionSchemas.submissionDtoSchema.parse(
        await service.delete(id)
      )

      return c.json({ submission })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not delete submission',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })
  .get('/:id', async (c) => {
    try {
      const id = +c.req.param('id')

      const submission = submissionSchemas.submissionDtoSchema.parse(
        await service.find(id)
      )

      return c.json({ submission })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find submission',
          code: HttpStatus.NOT_FOUND,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })
  .get('/', async (c) => {
    try {
      const submissions = submissionSchemas.submissionDtoSchema.array().parse(
        await service.findMany()
      )

      return c.json({ submissions })
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not find submissions',
          code: HttpStatus.NOT_FOUND,
          path: c.req.routePath,
          suggestion: 'check the input and try again',
        }),
        HttpStatus.NOT_FOUND
      )
    }
  })

        

