import { UserAchievementsService } from "@/services/user-achievements.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const associateSchema = z.object({
  userId: z.number(),
  achievementIds: z.array(z.number()),
});

const service = Container.get(UserAchievementsService);

export const userAchievementsRouter = honoWithJwt()
  .post('/associate', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { userId, achievementIds } = await c.req.valid('json');
        await service.associateUserWithAchievements(userId, achievementIds);
        return c.json({ message: 'Achievements associated successfully' });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate achievements',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/:userId/delete/achievement/:achievementId',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const achievementId = +c.req.param('achievementId');
        await service.removeAchievementsFromUser(userId, [achievementId]);
        return c.json({ message: 'Achievement removed successfully' });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to remove achievement',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/update', zValidator('json', associateSchema),
    async (c) => {
      try {
        const { userId, achievementIds } = await c.req.valid('json');
        await service.updateUserAchievements(userId, achievementIds);
        return c.json({ message: 'Achievements updated successfully' });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to update achievements',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

export const publicUserAchievementsRoute = new Hono()
  .get('/:userId/achievements',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const achievements = await service.getAchievementsByUser(userId);
        return c.json({ achievements });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to get user achievements',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .get('/:userId/achievements/:achievementId/exists',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const achievementId = +c.req.param('achievementId');
        const exists = await service.isAssociationExists(userId, achievementId);
        return c.json({ exists });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check if association exists',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .get('/achievements/:achievementId/users',
    async (c) => {
      try {
        const achievementId = +c.req.param('achievementId');
        const users = await service.getUsersByAchievement(achievementId);
        return c.json({ users });
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to get users by achievement',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )