import { UserCollectionsService } from "@/services/user-collection.service";
import Container from "typedi";
import { z } from "zod";
import { honoWithJwt } from "..";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { Hono } from "hono";

const associateSchema = z.object({
  userId: z.number(),
  collectionIds: z.array(z.number()),
})

const service = Container.get(UserCollectionsService);

export const userCollectionsRoute = honoWithJwt()
  .post('/associate', zValidator('json', associateSchema),
    async (c) => {
      try {
        const {userId, collectionIds } = await c.req.valid('json');
        await service.associateUserWithCollections(userId, collectionIds);
        return c.json({ message: 'Collections associates succesfully'}, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to associate collections',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the user and collection ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/:userId/delete/:collectionId', 
    async (c) => {
      try {
        const userid = +c.req.param('userId');
        const collectionId = +c.req.param('collectionId');
        await service.removeCollectionsFromUser(userid, [collectionId]);
        return c.json({ message: 'Collections removed succesfully'}, HttpStatus.OK);
      } catch (e) {
        return c.json (
          createApexError({ 
            status: 'error',
            message: 'Failed to remove collections',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the user and collection ids are valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

export const publicUserCollectionsRoutes = new Hono()
  .get('/:userId/collections',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const collections = await service.getCollectionByUser(userId);
        return c.json(collections, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch collections',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Ensure the user id is valid',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .get('/:userId/collections/:collectionId/exists',
    async (c) => {
      try {
        const userId = +c.req.param('userId');
        const collectionId = +c.req.param('collectionId');
        const exists = await service.isAssociationExists(userId, collectionId);
        return c.json( {exists}, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to check association',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
  .get('/collection/:collectionId/users',
    async (c) => {
      try {
        const collectionId = +c.req.param('collectionId');
        const users = await service.getUsersByCollection(collectionId);
        return c.json({ users }, HttpStatus.OK);
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'Failed to fetch users',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'Check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        );
      }
    }
  )
