import { honoWithJwt } from "..";
import Container from "typedi";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { userItemRelationSchemas } from "@/db/relations/user-item.relation";
import { z } from "zod";
import { UserService } from "@/services/user.service";
import { userSchemas, type UserProfile } from "@/db/schema/user.schema";
import { ItemsService } from "@/services/items.service";
import { itemsSchema, type ItemsModel } from "@/db/schema/item.schema";
import { UserItemRelationService } from "@/services/user-item.relation.service";

const service = Container.get(UserItemRelationService)
const userService = Container.get(UserService)
const itemService = Container.get(ItemsService)

export const userItemRouter = honoWithJwt()
  .post('/create', zValidator('json', userItemRelationSchemas.input),
    async (c) => {
      try{
        const input = await c.req.valid('json')
        let user = await userService.findById(input.user_id)
        let item = await itemService.findById(input.item_id) //tinha que ver questão de um usuario sem ser admin poder criar um admin
        let alreadyItem = await service.findByUserItem(input) != 0        
        if(user == null || item == null || alreadyItem){
          throw new Error()
        }

        const user_item = userItemRelationSchemas.dto.parse(
          await service.create(input)
        )
        return c.json({ user_item })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
  })
  .get('/items/user/:user_id', async (c) =>{ 
    try{
      const user_id = +c.req.param('user_id')
      const items_by_user = z.array(userItemRelationSchemas.model).parse(await service.findByUserId(user_id))
      var items: ItemsModel[] = []

      for(var element of items_by_user){
        let item = await itemService.findById(element.item_id)
        if(item != null)
          items.push(itemsSchema.dto.parse(item))
      }
      
      return c.json({ items })

    } catch(e){
      return c.notFound()
    }
  })
  .get('/users/item/:item_id', async (c) =>{
    try{
      const item_id = +c.req.param('item_id')
      const users_by_item = z.array(userItemRelationSchemas.model).parse(await service.findByItemId(item_id))
      
      var users: UserProfile[] = []

      for(var element of users_by_item){
        let user = await userService.findById(element.user_id)
        if(user != null)
          users.push(userSchemas.userProfileSchema.parse(user))
      }
      
      
      return c.json({ users })

    } catch(e){
      return c.notFound()
    }
  })
  .delete('/delete',
    zValidator('json', userItemRelationSchemas.input),
    async (c) =>{
      try{
        const input = await c.req.valid('json')
        const userItem_id = await service.findByUserItem(input)        
        const ret = userItemRelationSchemas.dto.parse(
          await service.delete(userItem_id)
        )
        return c.json(ret)
      } catch (e){
        return c.json(
          createApexError({
            status: 'error',
            message: 'could delete user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
  })