import { honoWithJwt } from "..";
import Container from "typedi";
import { zValidator } from "@hono/zod-validator";
import { createApexError, HttpStatus } from "@/services/error.service";
import { UserRoleRelationService } from "@/services/user-role.relation.service";
import { userRoleRelationSchemas } from "@/db/relations/user-role.relation";
import { z } from "zod";
import { UserService } from "@/services/user.service";
import { userSchemas, type UserProfile } from "@/db/schema/user.schema";
import { RoleService } from "@/services/role.service";
import { roleSchemas, type RoleModel } from "@/db/schema/role.schema";

const service = Container.get(UserRoleRelationService)
const userService = Container.get(UserService)
const roleService = Container.get(RoleService)

export const userRoleRouter = honoWithJwt()
  .post('/assign', zValidator('json', userRoleRelationSchemas.input),
    async (c) => {
      try{
        const input = await c.req.valid('json')
        const user = await userService.findById(input.user_id)
        const role = await roleService.findById(input.role_id) //tinha que ver questão de um usuario sem ser admin poder criar um admin
        const alreadyRole = await service.findByUserRole(input) != 0        
        if(user == null || role == null || alreadyRole){
          throw new Error()
        }

        const user_role = userRoleRelationSchemas.dto.parse(
          await service.create(input)
        )
        return c.json({ user_role })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not create user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
  })
  .get('/roles/user/:user_id', async (c) =>{ 
    try{
      const user_id = +c.req.param('user_id')
      const roles_by_user = z.array(userRoleRelationSchemas.model).parse(await service.findByUserId(user_id))
      const roles: RoleModel[] = []

      for(const element of roles_by_user){
        const role = await roleService.findById(element.role_id)
        if(role != null)
          roles.push(roleSchemas.dto.parse(role))
      }
      
      return c.json({ roles })

    } catch(e){
      return c.notFound()
    }
  })
  .get('/users/role/:role_id', async (c) =>{
    try{
      const role_id = +c.req.param('role_id')
      const users_by_role = z.array(userRoleRelationSchemas.model).parse(await service.findByRoleId(role_id))
      
      const users: UserProfile[] = []

      for(const element of users_by_role){
        const user = await userService.findById(element.user_id)
        if(user != null)
          users.push(userSchemas.userProfileSchema.parse(user))
      }
      
      
      return c.json({ users })

    } catch(e){
      return c.notFound()
    }
  })
  .post('/revoke',
    zValidator('json', userRoleRelationSchemas.input),
    async (c) =>{
      try{
        const input = await c.req.valid('json')
        const userRole_id = await service.findByUserRole(input)        
        const ret = userRoleRelationSchemas.dto.parse(
          await service.delete(userRole_id)
        )
        return c.json(ret)
      } catch (e){
        return c.json(
          createApexError({
            status: 'error',
            message: 'could delete user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
  })