import { UserStatsService } from "@/services/user-stats.service";
import { honoWithJwt } from "..";
import Container from "typedi";
import { zValidator } from "@hono/zod-validator";
import { userStatsSchemas } from "@/db/schema/user-stats.schema";
import { createApexError, HttpStatus } from "@/services/error.service";

const service = Container.get(UserStatsService)

export const userStatsRouter = honoWithJwt()
  /* Update é uma rota própria ou vai estar embutida nas rotas das devidas estatisticas */
  .post('/update',
    zValidator('json', userStatsSchemas.update),
    async (c) => {
      try {

        const input = await c.req.valid('json')
        const user_stats = userStatsSchemas.dto.parse(
          await service.update(input)
        )

        return c.json({ user_stats })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })

  /* Nao deveria ter como deletar um user stats sozinho */
  .post('/delete/:id',
    zValidator('json', userStatsSchemas.update),
    async (c) => {
      try {
        const id: number = +c.req.param('id')
        const user_stats = userStatsSchemas.dto.parse(
          await service.delete(id)
        )
        return c.json(user_stats)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could delete user stats',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })

  .post('/updateComments/:id',
    async (c) => {
      try {
        const id = +c.req.param('id')
        const stats = userStatsSchemas.dto.parse(await service.updateComments(id))

        return c.json(stats)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update the stats resource',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the id resource input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )

  /* Obtém todos os user stats */
  .get('/all', async (c) => {
    try {
      const stats = await service.getAll()
      return c.json(stats)
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not fetch user stats',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'try again later',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })

  /* Obtém um user stats específico */
  .get('/:id', async (c) => {
    try {
      const id = +c.req.param('id')
      const stat = await service.getById(id)

      if (!stat) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'user stats not found',
            code: HttpStatus.NOT_FOUND,
            path: c.req.routePath,
            suggestion: 'check the id and try again',
          }),
          HttpStatus.NOT_FOUND
        )
      }

      return c.json(stat)
    } catch (e) {
      return c.json(
        createApexError({
          status: 'error',
          message: 'could not fetch user stats',
          code: HttpStatus.BAD_REQUEST,
          path: c.req.routePath,
          suggestion: 'check the id and try again',
        }),
        HttpStatus.BAD_REQUEST
      )
    }
  })
