import { userSchemas, type UserProfile } from '@/db/schema/user.schema'
import { UserService } from '@/services/user.service'
import { Container } from 'typedi'
import { z } from 'zod'
import { zValidator } from '@hono/zod-validator'
import { honoWithJwt } from '..'
import { createApexError, HttpStatus } from '@/services/error.service'
import { followRelationSchemas } from '@/db/relations/followers.relation'
import { UserStatsService } from '@/services/user-stats.service'
import { FollowRelationService } from '@/services/follow.relation.service'

const service = Container.get(UserService)
const followService = Container.get(FollowRelationService)
const userStatsService = Container.get(UserStatsService)

export const userRouter = honoWithJwt()
  .post('/follow',
    zValidator('json', followRelationSchemas.input),
    async (c) => {
      try {
        const input = await c.req.valid('json')
        const follower = await userStatsService.getById(input.follower_id)
        const following = await userStatsService.getById(input.user_id)
        const alreadyFollow = await followService.findByFollow(input) != 0
        if (follower == null || following == null || input.follower_id == input.user_id || alreadyFollow) {
          throw new Error()
        }



        follower.follows++ // incremento do numero de pessoas que o seguidor segue
        following.followers++ // incremento do numero de seguidores que a pessoa seguida tem 

        await userStatsService.update(follower)
        await userStatsService.update(following)

        const followRelation = followRelationSchemas.dto.parse(
          await followService.create(input)
        )

        return c.json({ followRelation })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .post('/unfollow',
    zValidator('json', followRelationSchemas.input),
    async (c) => {
      try {
        const input = await c.req.valid('json')


        const follow = await followService.findByFollow(input)


        const follower = await userStatsService.getById(input.follower_id)
        const following = await userStatsService.getById(input.user_id)




        follower!.follows-- // decremento do numero de pessoas que o seguidor segue
        following!.followers-- // decremento do numero de seguidores que a pessoa seguida tem 

        await userStatsService.update(follower!)
        await userStatsService.update(following!)

        const followRelation = followRelationSchemas.dto.parse(
          await followService.delete(follow)
        )

        return c.json({ followRelation })
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    }
  )
  .get('/follows/:id', // check who this id follows
    async (c) => {
      try {
        const id = +c.req.param('id')
        const follows_id = z.array(followRelationSchemas.model).parse(await followService.findByUserId(id))
        const follows: UserProfile[] = []

        for (const element of follows_id) {
          const follow = await service.findById(element.follower_id)
          if (follow != null)
            follows.push(userSchemas.userProfileSchema.parse(follow))
        }

        return c.json({ follows })
      } catch (e) {
        return c.notFound()
      }
    })
  .get('/followers/:id', //check who follows this id
    async (c) => {
      try {
        const id = +c.req.param('id')
        const followers_id = z.array(followRelationSchemas.model).parse(await followService.findByFollowerId(id))
        const followers: UserProfile[] = []

        for (const element of followers_id) {
          const follow = await service.findById(element.follower_id)
          if (follow != null)
            followers.push(userSchemas.userProfileSchema.parse(follow))
        }

        return c.json({ followers })

      } catch (e) {
        return c.notFound()
      }
    })
  .get('/users', async (c) => {
    const p = c.get('jwtPayload')

    console.log({ p })

    const users = await service.findMany()

    const ret = z.array(userSchemas.userDtoSchema).parse(users)

    return c.json({ users: ret })
  })
  .get('/:username', async (c) => {
    try {
      const username = c.req.param('username')
      const user = await service.findByUsername(username)
      const ret = userSchemas.userDtoSchema.parse(user)

      return c.json({ user: ret })
    } catch {
      return c.notFound()
    }
  })
  .post('/update',
    zValidator('json', userSchemas.userUpdateSchema),
    async (c) => {
      try {
        const input = await c.req.valid('json')

        const user = userSchemas.userDtoSchema.parse(
          await service.update(input)
        )

        return c.json({ user })
      } catch (e) {

        return c.json(
          createApexError({
            status: 'error',
            message: 'could not update user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )

      }
    })
  .post('/confirmation/:email',
    async (c) => {
      try {
        const email: string = c.req.param('email')
        const user = await service.findByEmail(email)
        if (user == null)
          throw new Error()

        user.confirmed_at = service.getUpdateTime()

        const confirmed_user = userSchemas.userDtoSchema.parse(
          await service.update(user)
        )

        return c.json(confirmed_user)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not delete user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })
  .post('/reactivate/:email',
    async (c) => {
      try {
        const email: string = c.req.param('email')
        const user = await service.findByEmail(email)
        if (user == null)
          throw new Error()

        user.active = true

        user.reactivated_at = service.getUpdateTime()

        const reactivated_user = userSchemas.userDtoSchema.parse(
          await service.update(user)
        )

        return c.json(reactivated_user)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not delete user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })
  .post('/delete/:id',
    async (c) => {
      try {
        const id: number = +c.req.param('id')
        const user = await service.findById(id)
        if (user == null)
          throw new Error()

        user.active = false

        user.deleted_at = service.getUpdateTime()

        const deleted_user = userSchemas.userDtoSchema.parse(
          await service.update(user)
        )

        return c.json(deleted_user)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not delete user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })
  .post('/delete/system/:id',
    async (c) => {
      try {
        const id: number = +c.req.param('id')
        const user = userSchemas.userDtoSchema.parse(
          await service.systemDelete(id)
        )

        return c.json(user)
      } catch (e) {
        return c.json(
          createApexError({
            status: 'error',
            message: 'could not delete user',
            code: HttpStatus.BAD_REQUEST,
            path: c.req.routePath,
            suggestion: 'check the input and try again',
          }),
          HttpStatus.BAD_REQUEST
        )
      }
    })  
