/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const dailyChargeAmountApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const id2str = require(`${libs}/middlewares/id2str`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const request = require(`request`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: { include: [200] } }).middleware;

let rqf = new ReqQueryFields();

let rqfCount = new ReqQueryFields();

//dailyChargeAmountApp.use(cache('15 day'));

dailyChargeAmountApp.get('/year_range', (req, res, next) => {
    req.sql.from('turma')
        .field('MIN(turma.ano_censo)', 'start_year')
        .field('MAX(turma.ano_censo)', 'end_year');
    next();
}, query, response('range'));

dailyChargeAmountApp.get('/years', (req, res, next) => {
    req.sql.from('turma')
        .field('DISTINCT turma.ano_censo', 'year');
    next();
}, query, response('years'));

dailyChargeAmountApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
        .field('fonte', 'source')
        .where('tabela = \'turma\'');
    next();
}, query, response('source'));

dailyChargeAmountApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

dailyChargeAmountApp.get('/adm_dependency_detailed', cache('15 day'), (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 6; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

dailyChargeAmountApp.get('/location', cache('15 day'), (req, res, next) => {
    req.result = [
        { id: 1, name: 'Urbana' },
        { id: 2, name: 'Rural' }
    ];
    next();
}, response('location'));

dailyChargeAmountApp.get('/rural_location', (req, res, next) => {
    req.result = [
        { id: 1, name: "Urbana" },
        { id: 2, name: "Rural" },
        { id: 3, name: "Rural - Área de assentamento" },
        { id: 4, name: "Rural - Terra indígena" },
        { id: 5, name: "Rural - Área remanescente de quilombos" },
        { id: 6, name: "Rural - Unidade de uso sustentável" }
    ];
    next();
}, response('rural_location'));

dailyChargeAmountApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        { id: null, name: 'Não classificada' },
        { id: 1, name: 'Creche' },
        { id: 2, name: 'Pré-Escola' },
        { id: 3, name: 'Ensino Fundamental - anos iniciais' },
        { id: 4, name: 'Ensino Fundamental - anos finais' },
        { id: 5, name: 'Ensino Médio' },
        { id: 6, name: 'EJA' },
        { id: 7, name: 'EE exclusiva' }
    ];
    next();
}, response('education_level_short'));

dailyChargeAmountApp.get('/average/education_level_mod', (req, res, next) => {
    req.result = [
        { id: null, name: 'Não classificada' },
        { id: 1, name: 'Creche' },
        { id: 2, name: 'Pré-Escola' },
        { id: 3, name: 'Educação Infantil Unificada' },
        { id: 4, name: 'Ensino Fundamental - anos iniciais' },
        { id: 5, name: 'Ensino Fundamental - anos finais' },
        { id: 6, name: 'Ensino Médio' },
        { id: 7, name: 'Turma Multisseriadas e Multietapas' },
        { id: 8, name: 'EJA - Ensino Fundamental' },
        { id: 9, name: 'EJA - Ensino Médio' },
        { id: 10, name: 'Educação Profissional' }
    ];
    next();
}, response('education_level_mod'));

dailyChargeAmountApp.get('/period', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 3; ++i) {
        req.result.push({
            id: i,
            name: id2str.period(i)
        });
    };
    req.result.push({
        id: 99,
        name: id2str.period(99)
    });
    next();
}, response('period'));

dailyChargeAmountApp.get('/integral_time', (req, res, next) => {
    req.result = [
        { id: null, name: 'Não Disponível' },
        { id: 0, name: 'Não' },
        { id: 1, name: 'Sim' }
    ];
    next();
}, response('integral_time'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'turma'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'min_year',
    table: 'turma',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: 'turma',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'turma',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: 'turma',
        field: 'ano_censo'
    }
}).addValue({
    name: 'location',
    table: 'turma',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'adm_dependency',
    table: 'turma',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'turma',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'rural_location',
    table: 'turma',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'turma',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    dontGroup: true,
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
    name: 'period',
    table: 'turma',
    tableField: 'turma_turno_id',
    resultField: 'period_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'turma_turno_id'
    }
}).addValue({
    name: 'integral_time',
    table: 'turma',
    tableField: 'tempo_integral',
    resultField: 'integral_time_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'tempo_integral'
    }
}).addValue({
    name: 'education_level_short',
    table: 'turma',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
});

dailyChargeAmountApp.get('/', rqf.parse(), (req, res, next) => {
    var status = 0;
    if ('period' in req.filter) {
        if (req.filter['period'].length == 1 && (req.filter['period'][0] == '3' || req.filter['period'][0] === '4')) {
            status = 1;
        } else if (req.filter['period'].length <= 2 && (req.filter['period'].includes('1') || req.filter['period'].includes('2')) && (!req.filter['period'].includes('3'))) {
            status = 1;
        }
    } else if (req.filter['integral_time'] == '1') {
        status = 1;
    }

    if (status) {
        req.sql.from('turma')
            .field('turma.ano_censo', 'year')
            .field('turma.etapa_resumida', 'education_level_short_id')
            .field('AVG(turma.duracao_turma)/60.0', 'average_class_duration')
            .field('MEDIAN(turma.duracao_turma)/60.0', 'median_class_duration')
            .field('STDDEV_SAMP(turma.duracao_turma)/60.0', 'std_class_duration')
            .field('QUANTILE(turma.duracao_turma, 0.25)/60.0', 'fstqt_class_duration')
            .field('QUANTILE(turma.duracao_turma, 0.75)/60.0', 'thdqt_class_duration')
            .group('turma.ano_censo')
            .group('turma.etapa_resumida')
            .order('turma.ano_censo')
            .order('turma.etapa_resumida')
            .where('((turma.tipo_turma_id <= 3 AND turma.tipo_atendimento_id is NULL) OR (turma.tipo_atendimento_id <= 2 AND turma.tipo_turma_id is NULL)) and turma.dependencia_adm_id <= 3')
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }

    next();

}, rqf.build(), query, addMissing(rqf), (req, res, next) => {

    function sliced(object) {
        return object['education_level_short_id'] > 3;
    }

    if ('period' in req.filter || 'period' in req.dims) {
        req.filter['period'].forEach((element) => {
            if (element == '3')
                req.result = req.result.filter(sliced);
        });
    }

    next();
}, id2str.transform(), response('turma'));

dailyChargeAmountApp.get('/average', rqf.parse(), rqf.build(), (req, res, next) => {
    var status = 0;
    if (('education_level_mod' in req.filter || 'education_level_mod' in req.dims)
        && ('integral_time' in req.filter)) {
        if (req.filter['integral_time'] == '0'
            && ('period' in req.filter)) {
            if (req.filter['period'].length == 1
                && req.filter['period'][0] == '3') {
                status = 1;
            } else if (req.filter['period'].length <= 2
                && (req.filter['period'].includes('1')
                    || req.filter['period'].includes('2'))
                && (!req.filter['period'].includes('3'))) {
                status = 1;
            }
        } else if (req.filter['integral_time'] == '1') {
            status = 1;
        }
    }

    if (status) {
        let baseQ = req.sql.clone();

        let tableR = baseQ.clone();
        tableR.from('turma')
            .field('duracao_turma')
            .field('etapas_mod_ensino_segmento_id')
            .field('ROW_NUMBER() OVER(PARTITION BY etapas_mod_ensino_segmento_id ORDER BY duracao_turma)', 'rowno')
            .where('((turma.tipo_turma_id <= 3 AND turma.tipo_atendimento_id is NULL) OR (turma.tipo_atendimento_id <= 2 AND turma.tipo_turma_id is NULL))')

        let tableG = baseQ.clone();
        tableG.from('turma')
            .field('1+COUNT(*)', 'counter')
            .field('etapas_mod_ensino_segmento_id')
            .where('((turma.tipo_turma_id <= 3 AND turma.tipo_atendimento_id is NULL) OR (turma.tipo_atendimento_id <= 2 AND turma.tipo_turma_id is NULL))')
            .group('etapas_mod_ensino_segmento_id')

        let joinRG = squel.select();
        joinRG.from(tableR, 'R')
            .field('R.etapas_mod_ensino_segmento_id')
            .field('AVG(1.0*R.duracao_turma)/60', 'median_value')
            .join(tableG, 'G', 'R.etapas_mod_ensino_segmento_id = G.etapas_mod_ensino_segmento_id AND R.rowNo BETWEEN G.counter/2 AND G.counter/2+G.counter%2')
            .group('R.etapas_mod_ensino_segmento_id')

        req.sql
            .from('turma')
            .from(joinRG, 'm')
            .field('turma.ano_censo', 'year')
            .field('turma.etapas_mod_ensino_segmento_id', 'education_level_mod_id')
            .field('AVG(turma.duracao_turma)/60.0', 'average_class_duration')
            .field('AVG(m.median_value)', 'median_class_duration')
            .field('STDDEV_SAMP(turma.duracao_turma)/60.0', 'std_class_duration')
            .field('QUANTILE(turma.duracao_turma, 0.25)/60.0', 'fstqt_class_duration')
            .field('QUANTILE(turma.duracao_turma, 0.75)/60.0', 'thdqt_class_duration')
            .group('turma.ano_censo')
            .group('turma.etapas_mod_ensino_segmento_id')
            .order('turma.ano_censo')
            .order('turma.etapas_mod_ensino_segmento_id')
            .where('((turma.tipo_turma_id <= 3 AND turma.tipo_atendimento_id is NULL) OR (turma.tipo_atendimento_id <= 2 AND turma.tipo_turma_id is NULL)) \
                and m.etapas_mod_ensino_segmento_id = turma.etapas_mod_ensino_segmento_id')
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }

    next();
}, query, addMissing(rqf), id2str.transform(), response('turma'));

module.exports = dailyChargeAmountApp;
