/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const disciplinesApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: { include: [200] } }).middleware;

let rqf = new ReqQueryFields();

disciplinesApp.use(cache('15 day'));

// Returns a tuple of start and ending years of the complete enrollments dataset.
disciplinesApp.get('/year_range', (req, res, next) => {
    req.sql.from('docente')
        .field('MIN(docente.ano_censo)', 'start_year')
        .field('MAX(docente.ano_censo)', 'end_year');
    next();
}, query, response('range'));

disciplinesApp.get('/years', (req, res, next) => {
    req.sql.from('docente').
        field('DISTINCT docente.ano_censo', 'year');
    next();
}, query, response('years'));

disciplinesApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
        .field('fonte', 'source')
        .where('tabela = \'docente\'');
    next();
}, query, response('source'));

disciplinesApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 8; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

disciplinesApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

disciplinesApp.get('/education_level_mod', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 12; ++i) {
        req.result.push({
            id: i,
            name: id2str.educationLevelMod(i)
        });
    }
    req.result.push({
        id: 99,
        name: id2str.educationLevelMod(99)
    });
    next();
}, response('education_level_mod'));

disciplinesApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        { id: null, name: 'Não classificada' },
        { id: 1, name: 'Creche' },
        { id: 2, name: 'Pré-Escola' },
        { id: 3, name: 'Ensino Fundamental - anos iniciais' },
        { id: 4, name: 'Ensino Fundamental - anos finais' },
        { id: 5, name: 'Ensino Médio' },
        { id: 6, name: 'EJA' },
        { id: 7, name: 'EE exclusiva' }
    ];
    next();
}, response('education_level_short'));

disciplinesApp.get('/location', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.location(i)
        });
    };
    next();
}, response('location'));

disciplinesApp.get('/rural_location', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 8; i++) {
        req.result.push({
            id: i,
            name: id2str.ruralLocation(i)
        });
    };
    next();
}, response('rural_location'));

disciplinesApp.get('/education_type', (req, res, next) => {
    req.sql.from('docente')
        .field('DISTINCT nivel_tipo_formacao', 'id')
        .order('id');
    next();
}, query, (req, res, next) => {
    req.result.forEach((result) => {
        result.name = id2str.educationType(result.id);
    });
    next();
}, response('education_type'));

disciplinesApp.get('/gender', (req, res, next) => {
    req.result = [
        { id: 1, name: 'Masculino' },
        { id: 2, name: 'Feminino' }
    ];
    next();
}, response('gender'));


disciplinesApp.get('/contract_type', (req, res, next) => {
    req.result = [{
        id: "null",
        name: id2str.contractType("null")
    }];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.contractType(i)
        });
    }
    next();
}, response('contract_type'));

disciplinesApp.get('/ethnic_group', (req, res, next) => {
    req.result = [];
    for (let i = 0; i <= 5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroup(i)
        });
    }
    next();
}, response('ethnic_group'));

disciplinesApp.get('/discipline', (req, res, next) => {
    req.result = [];
    for (let i = 1; i <= 19; i++) {
        req.result.push({
            id: i,
            name: id2str.discipline(i)
        })
    }
    next();
}, response('discipline'))

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'adm_dependency',
    table: 'docente',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'docente',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'contract_type',
    table: 'docente',
    tableField: 'tipo_contratacao',
    resultField: 'contract_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tipo_contratacao'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'docente',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
    name: 'education_level_short',
    table: 'docente',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
}).addValue({
    name: 'education_type',
    table: 'docente',
    tableField: 'nivel_tipo_formacao',
    resultField: 'education_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_tipo_formacao'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_regiao_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'mesoregion',
    table: 'municipio',
    tableField: ['nome_mesorregiao', 'mesorregiao_id'],
    resultField: ['mesoregion_name', 'mesoregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'microregion',
    table: 'municipio',
    tableField: ['nome_microrregiao', 'microrregiao_id'],
    resultField: ['microregion_name', 'microregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_estado_id',
        foreignTable: 'docente'
    }
}).addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'filter').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: 'nome_escola',
    resultField: 'school_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'filter').addValue({
    name: 'location',
    table: 'docente',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'rural_location',
    table: 'docente',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'min_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'gender',
    table: 'docente',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'ethnic_group',
    table: 'docente',
    tableField: 'cor_raca',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'discipline',
    table: 'docente',
    tableField: '',
    resultField: '',
    where: {
        relation: '=',
        type: 'integer',
        field: ''
    }
});

disciplinesApp.get('/', rqf.parse(), (req, res, next) => {
    if ('discipline' in req.dims) {
        // delete req.filter.discipline;
        delete req.dims.discipline;
        req.tmp_discipline = true;

        req.sql.field('SUM(n_disc)', 'total')
            .field('SUM(n_disc_adequada)', 'total_suitable')

            .field('SUM(quimica_not_null)', 'total_quimica')
            .field('SUM(adequacao_quimica)', 'total_suitable_quimica')

            .field('SUM(fisica_not_null)', 'total_fisica')
            .field('SUM(adequacao_fisica)', 'total_suitable_fisica')

            .field('SUM(matematica_not_null)', 'total_matematica')
            .field('SUM(adequacao_matematica)', 'total_suitable_matematica')

            .field('SUM(biologia_not_null)', 'total_biologia')
            .field('SUM(adequacao_biologia)', 'total_suitable_biologia')

            .field('SUM(ciencias_not_null)', 'total_ciencias')
            .field('SUM(adequacao_ciencias)', 'total_suitable_ciencias')

            .field('SUM(lingua_portuguesa_not_null)', 'total_lingua_portuguesa')
            .field('SUM(adequacao_lingua_portuguesa)', 'total_suitable_lingua_portuguesa')

            .field('SUM(lingua_inglesa_not_null)', 'total_lingua_inglesa')
            .field('SUM(adequacao_lingua_inglesa)', 'total_suitable_lingua_inglesa')

            .field('SUM(lingua_espanhola_not_null)', 'total_lingua_espanhola')
            .field('SUM(adequacao_lingua_espanhola)', 'total_suitable_lingua_espanhola')

            .field('SUM(lingua_francesa_not_null)', 'total_lingua_francesa')
            .field('SUM(adequacao_lingua_francesa)', 'total_suitable_lingua_francesa')

            .field('SUM(lingua_outra_not_null)', 'total_lingua_outra')
            .field('SUM(adequacao_lingua_outra)', 'total_suitable_lingua_outra')

            .field('SUM(lingua_indigena_not_null)', 'total_lingua_indigena')
            .field('SUM(adequacao_lingua_indigena)', 'total_suitable_lingua_indigena')

            .field('SUM(artes_not_null)', 'total_artes')
            .field('SUM(adequacao_artes)', 'total_suitable_artes')

            .field('SUM(educacao_fisica_not_null)', 'total_educacao_fisica')
            .field('SUM(adequacao_educacao_fisica)', 'total_suitable_educacao_fisica')

            .field('SUM(historia_not_null)', 'total_historia')
            .field('SUM(adequacao_historia)', 'total_suitable_historia')

            .field('SUM(geografia_not_null)', 'total_geografia')
            .field('SUM(adequacao_geografia)', 'total_suitable_geografia')

            .field('SUM(filosofia_not_null)', 'total_filosofia')
            .field('SUM(adequacao_filosofia)', 'total_suitable_filosofia')

            .field('SUM(ensino_religioso_not_null)', 'total_ensino_religioso')
            .field('SUM(adequacao_ensino_religioso)', 'total_suitable_ensino_religioso')

            .field('SUM(estudos_sociais_not_null)', 'total_estudos_sociais')
            .field('SUM(adequacao_estudos_sociais)', 'total_suitable_estudos_sociais')

            .field('SUM(sociologia_not_null)', 'total_sociologia')
            .field('SUM(adequacao_sociologia)', 'total_suitable_sociologia')

            .field("'Brasil'", 'name')
            .field('docente.ano_censo', 'year')
            .from('docente')
            .group('docente.ano_censo')
            .order('docente.ano_censo')
            .where('(docente.tipo_docente = 1 OR docente.tipo_docente = 5) AND \
            ((docente.tipo_turma_id >= 0 AND docente.tipo_turma_id <= 3 AND docente.tipo_turma_atendimento_id is NULL) \
            OR ((docente.tipo_turma_atendimento_id = 1 OR docente.tipo_turma_atendimento_id = 2) AND docente.tipo_turma_id is NULL)) AND \
            docente.etapas_mod_ensino_segmento_id <> 6 AND docente.etapas_mod_ensino_segmento_id <> 12');
    }
    else if ('discipline' in req.filter) {
        const disciplines = ['quimica', 'fisica', 'matematica', 'biologia', 'ciencias', 'lingua_portuguesa', 'lingua_inglesa', 'lingua_espanhola', 'lingua_francesa', 'lingua_outra', 'lingua_indigena', 'artes', 'educacao_fisica', 'historia', 'geografia', 'filosofia', 'ensino_religioso', 'estudos_sociais', 'sociologia']

        let totalQuery = 'SUM('
        let totalSuitableQuery = 'SUM('

        if (!(req.filter.discipline instanceof Array)) req.filter.discipline = [req.filter.discipline]

        req.filter.discipline.forEach(d => {
            totalQuery += disciplines[d - 1] + '_not_null + '
            totalSuitableQuery += 'adequacao_' + disciplines[d - 1] + ' + '
        })
        totalQuery = totalQuery.slice(0, -2) + ')'
        totalSuitableQuery = totalSuitableQuery.slice(0, -2) + ')'

        delete req.filter.discipline;

        req.sql.field(totalQuery, 'total')
            .field(totalSuitableQuery, 'total_suitable')
            .field("'Brasil'", 'name')
            .field('docente.ano_censo', 'year')
            .from('docente')
            //    .join('turma', null, 'docente.turma_id=turma.id AND docente.ano_censo=turma.ano_censo')
            .group('docente.ano_censo')
            .order('docente.ano_censo')
            .where('(docente.tipo_docente = 1 OR docente.tipo_docente = 5) AND \
                ((docente.tipo_turma_id >= 0 AND docente.tipo_turma_id <= 3 AND docente.tipo_turma_atendimento_id is NULL) \
                OR ((docente.tipo_turma_atendimento_id = 1 OR docente.tipo_turma_atendimento_id = 2) AND docente.tipo_turma_id is NULL)) AND \
                docente.etapas_mod_ensino_segmento_id <> 6 AND docente.etapas_mod_ensino_segmento_id <> 12');
    }
    else {
        req.sql.field('SUM(n_disc)', 'total')
            .field('SUM(n_disc_adequada)', 'total_suitable')
            .field("'Brasil'", 'name')
            .field('docente.ano_censo', 'year')
            .from('docente')
            //    .join('turma', null, 'docente.turma_id=turma.id AND docente.ano_censo=turma.ano_censo')
            .group('docente.ano_censo')
            .order('docente.ano_censo')
            .where('(docente.tipo_docente = 1 OR docente.tipo_docente = 5) AND \
            ((docente.tipo_turma_id >= 0 AND docente.tipo_turma_id <= 3 AND docente.tipo_turma_atendimento_id is NULL) \
            OR ((docente.tipo_turma_atendimento_id = 1 OR docente.tipo_turma_atendimento_id = 2) AND docente.tipo_turma_id is NULL)) AND \
            docente.etapas_mod_ensino_segmento_id <> 6 AND docente.etapas_mod_ensino_segmento_id <> 12');
    }

    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    let filters = Object.keys(req.filter)
	
	// if(filters.includes("state")) {
	// 	const disciplinesDB = ['quimica', 'fisica', 'matematica', 'biologia', 'ciencias', 'lingua_portuguesa', 'lingua_inglesa', 'lingua_espanhola',
	// 		'lingua_francesa', 'lingua_outra', 'lingua_indigena', 'artes', 'educacao_fisica', 'historia', 'geografia', 'filosofia', 'ensino_religioso',
	// 		'estudos_sociais', 'sociologia']
	// 	const disciplinesAPI = ["Química", "Física", "Matemática", "Biologia", "Ciências", "Língua Portuguesa", "Língua Estrangeira – Inglês",
	// 		"Língua Estrangeira - Espanhol","Língua Estrangeira - Francês", "Língua Estrangeira - Outras", "Língua Indígena", "Arte", "Educação Física", "História",
	// 		"Geografia", "Filosofia", "Ensino religioso", "Estudos Sociais", "Sociologia"]
	
	// 	let jsonKeys = []
	// 	let results = []
	// 	req.result.forEach((r) => {
	// 		jsonKeys = Object.keys(r)
		
	// 		let i
	// 		let size = jsonKeys.length - 2  // Last two infos are "name" and "year"
	// 		for(i = 0; i < size; i++) {
	// 			let total_name = jsonKeys[i]
	// 			let suitable_name = jsonKeys[i + 1]
			
	// 			// Calculates percentage
	// 			let percentage = r[suitable_name] / r[total_name]
	// 			percentage = percentage * 100
	// 			percentage = percentage.toFixed(1)	// Rounds to 1 digit after comma, returns string
	// 			percentage = percentage.replace(".", ",") + "%"

	// 			// Parses name
	// 	        total_name = total_name.replace("total_", "")
	// 			let discipline_index = disciplinesDB.indexOf(total_name)
	// 			let discipline_name = disciplinesAPI[discipline_index]

	// 			let obj = {
	// 				total: percentage,
	// 				name: r["name"],
	// 				year: r["year"],
	// 				discipline_id: discipline_index + 1,	// Convert function starts at 1, not at 0
	// 				discipline_name: discipline_name
	// 			}
	// 			results.push(obj)
			
	// 			i++;    // Ignore next, it's a suitable already used
	// 		}
	// 	})

	// 	req.result = results;
	// }
	// else {
		let disciplinesNotSuitable = [];
		let disciplinesSuitable = [];

		req.result.forEach((r) => {

            let obj = {
                sum_total: 0,
                sum_suitable: 0
            }

	        Object.keys(r).forEach(k => {
                if (k !== 'total' && k !== 'total_suitable') 
                    obj[k] = r[k];
			})

            if (req.tmp_discipline){
                Object.keys(r).forEach(k => {
                    if (/^total_suitable_/.test(k)) // if k starts with total_suitable
                        obj.sum_suitable += parseInt(r[k]);
                    else if (/^total_(?!suitable)/.test(k))
                        obj.sum_total += parseInt(r[k]);
                })
            } else {
                delete obj.sum_total;
                delete obj.sum_suitable;
            }

            let objNotSuitable = Object.assign({}, {
				total: parseInt(r.total) - parseInt(r.total_suitable),
				suitable: 0,
                discipline_name: 'Formação não adequada',
			}, obj)

	        let objSuitable = Object.assign({}, {
		        total: parseInt(r.total_suitable),
			    suitable: 1,
                discipline_name: 'Formação adequada',
            }, obj)

		    disciplinesNotSuitable.push(objNotSuitable)
			disciplinesSuitable.push(objSuitable)
		 })

		 req.result = disciplinesNotSuitable.concat(disciplinesSuitable);
    next();
}, response('disciplines'));

module.exports = disciplinesApp;


