/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const classApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqfCount = new ReqQueryFields();

classApp.use(cache('15 day'));

// Complete range of the enrollments dataset.
// Returns a tuple of start and ending years of the complete enrollments dataset.
classApp.get('/year_range', (req, res, next) => {
    req.sql.from('turma')
    .field('MIN(turma.ano_censo)', 'start_year')
    .field('MAX(turma.ano_censo)', 'end_year');
    next();
}, query, response('range'));

classApp.get('/years', (req, res, next) => {
    req.sql.from('turma')
    .field('DISTINCT turma.ano_censo', 'year');
    next();
}, query, response('years'));

classApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'turma\'');
    next();
}, query, response('source'));

classApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

classApp.get('/diff_location', (req, res, next) => {
    req.result = [
        {id: 0, name: "A escola não está em localidade diferenciada"},
        {id: 1, name: "Área de assentamento"},
        {id: 2, name: "Terra indígena"},
        {id: 3, name: "Terra remanescente de quilombos"},
    ];
    next();
}, response('diff_location'));

// Returns all adm dependencies
classApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

classApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 8; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

// Returns all periods avaible
classApp.get('/period', (req, res, next) => {
    req.result = [];
    for(let i=1; i <= 3; ++i) {
        req.result.push({
            id: i,
            name: id2str.period(i)
        });
    }
    req.result.push({
        id: 99,
        name: id2str.period(99)
    });
    next();
}, response('period'));

// Returns integral-time avaible
classApp.get('/integral_time', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.integralTime(i)
        });
    }
    next();
}, response('integral_time'));

// Returns all educational levels avaible
classApp.get('/education_level_mod', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <=12; ++i) {
        req.result.push({
            id: i,
            name: id2str.educationLevelMod(i)
        });
    }
    next();
}, response('education_level_mod'));

classApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Creche'},
        {id: 2, name: 'Pré-Escola'},
        {id: 3, name: 'Ensino Fundamental - anos iniciais'},
        {id: 4, name: 'Ensino Fundamental - anos finais'},
        {id: 5, name: 'Ensino Médio'},
        {id: 6, name: 'EJA'},
        {id: 7, name: 'EE exclusiva'}
    ];
    next();
}, response('education_level_short'));

rqfCount.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'turma'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'turma'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'turma'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'turma'
    }
}, 'filter').addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'estado_id',
        table: 'turma'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'turma'
    }
}).addValue({
    name: 'min_year',
    table: 'turma',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'turma',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name:'adm_dependency',
    table: 'turma',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'location',
    table: 'turma',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'diff_location',
    table: 'turma',
    tableField: 'localizacao_diferenciada_par',
    resultField: 'diff_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_diferenciada_par'
    }
}).addValue({
    name: 'rural_location',
    table: 'turma',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
  name:'education_level_mod',
  table: 'turma',
  tableField: 'etapas_mod_ensino_segmento_id',
  resultField: 'education_level_mod_id',
  where: {
    relation: '=',
    type: 'integer',
    field: 'etapas_mod_ensino_segmento_id'
  }
}).addValue({
    name: 'education_level_short',
    table: 'turma',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'turma',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
  name:'period',
  table: 'turma',
  tableField: 'turma_turno_id',
  resultField: 'period_id',
  where: {
      relation: '=',
      type: 'integer',
      field: 'turma_turno_id'
  }
}).addValue({
  name:'integral_time',
  table: 'turma',
  tableField: 'tempo_integral',
  resultField: 'integral_time_id',
  where: {
      relation: '=',
      type: 'integer',
      field: 'tempo_integral'
  }
}).addValueToField({
    name: 'school',
    table: 'escola_agregada',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'turma'
    }
}, 'dims').addValueToField({
    name: 'school',
    table: 'escola_agregada',
    tableField: 'nome_escola',
    resultField: 'school_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'turma'
    }
}, 'filter').addValueToField({
    name: 'period_not',
    table: 'turma',
    tableField: 'turma_turno_id',
    resultField: 'period_id',
    where: {
        relation: '<>',
        type: 'integer',
        field: 'turma_turno_id'
    }
}, 'filter');


classApp.get('/', rqfCount.parse(), (req, res, next) => {
    req.sql.field('COUNT(turma.id)', 'total')
    .field("'Brasil'", 'name')
    .field('turma.ano_censo', 'year')
    .from('turma')
    .group('turma.ano_censo')
    .order('turma.ano_censo')
    .where('(turma.tipo_turma_id >= 0 AND turma.tipo_turma_id <= 3 AND turma.tipo_atendimento_id is NULL) \
        OR ((turma.tipo_atendimento_id = 1 OR turma.tipo_atendimento_id = 2) AND turma.tipo_turma_id is NULL) ');
    next();
}, rqfCount.build(), query, addMissing(rqfCount), id2str.transform(), response('class'));

module.exports = classApp;
