/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const infrastructureApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const response = require(`${libs}/middlewares/response`);

const id2str = require(`${libs}/middlewares/id2str`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

infrastructureApp.use(cache('15 day'));

infrastructureApp.get('/year_range', (req, res, next) => {
    req.sql.from('escola_agregada')
    .field('MIN(escola_agregada.ano_censo)', 'start_year')
    .field('MAX(escola_agregada.ano_censo)', 'end_year');
    next();
}, query, response('range'));

infrastructureApp.get('/years', (req, res, next) => {
    req.sql.from('escola_agregada')
    .field('DISTINCT escola_agregada.ano_censo', 'year');
    next();
}, query, response('years'));

infrastructureApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'escola_agregada\'');
    next();
}, query, response('source'));

infrastructureApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

infrastructureApp.get('/rural_location', (req, res, next) => {
    req.result = [
        {id: 1, name: "Urbana"},
        {id: 2, name: "Rural"},
        {id: 3, name: "Rural - Área de assentamento"},
        {id: 4, name: "Rural - Terra indígena"},
        {id: 5, name: "Rural - Área remanescente de quilombos"},
        {id: 6, name: "Rural - Unidade de uso sustentável"}
    ];
    next();
}, response('rural_location'));

infrastructureApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

infrastructureApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 6; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'escola_agregada'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'escola_agregada'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'id',
    resultField: 'city_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'escola_agregada'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'escola_agregada'
    }
}, 'filter').addValueToField({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'estado_id',
        table: 'escola_agregada'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'escola_agregada'
    }
}, 'dims').addValueToField({
    name: 'state',
    table: 'estado',
    tableField: 'id',
    resultField: 'state_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'estado_id',
        table: 'escola_agregada'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'escola_agregada'
    }
}, 'filter').addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'escola_agregada'
    }
}).addValue({
    name: 'location',
    table: 'escola_agregada',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'rural_location',
    table: 'escola_agregada',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'adm_dependency',
    table: 'escola_agregada',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'escola_agregada',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'min_year',
    table: 'escola_agregada',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'escola_agregada',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
});

function matchQueries(queryTotal, queryPartial) {
    let match = [];
    queryTotal.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryPartial.length; ++i) {
            let partial = queryPartial[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(partial[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = partial;
                break;
            }
        }

        if(objMatch) {
            newObj.percentage = (objMatch.total / result.total) * 100;
            newObj.partial = objMatch.total;
            newObj.total = result.total
            match.push(newObj);
        }
    });

    return match;
}

infrastructureApp.get('/', rqf.parse(), rqf.build(), (req, res, next) => {
    req.querySet = [];
    req.queryIndex = {};

    // Local de funcionamento
    let allSchools = req.sql.clone();
    allSchools.from('escola_agregada').field('COUNT(escola_agregada.id)', 'total')
    .field("'Brasil'", 'name')
    .field('escola_agregada.ano_censo', 'year')
    .group('escola_agregada.ano_censo')
    .where('escola_agregada.situacao_de_funcionamento = 1')
    .order('escola_agregada.ano_censo');
    req.queryIndex.allSchools = req.querySet.push(allSchools) - 1;

    let schoolPlace = allSchools.clone();
    schoolPlace.where('escola_agregada.func_predio_escolar = 1 AND escola_agregada.func_salas_empresa = 0 AND escola_agregada.func_templo_igreja = 0 AND escola_agregada.func_casa_professor = 0 AND escola_agregada.func_galpao = 0 AND escola_agregada.biblioteca = 1');
    req.queryIndex.schoolPlace = req.querySet.push(schoolPlace) - 1;

    // Bibliotecas
    let allLibraries = allSchools.clone();
    allLibraries.where('escola_agregada.func_predio_escolar = 1 AND escola_agregada.localizacao_id = 1');
    req.queryIndex.allLibraries = req.querySet.push(allLibraries) - 1;

    let haveLibraries = allLibraries.clone();
    haveLibraries.where('escola_agregada.biblioteca = 1');
    req.queryIndex.haveLibraries = req.querySet.push(haveLibraries) - 1;

    // Bibliotecas/Sala de leitura
    let allLibrariesReadingRoom = allSchools.clone();
    allLibrariesReadingRoom.where('escola_agregada.func_predio_escolar = 1 AND escola_agregada.localizacao_id = 2');
    req.queryIndex.allLibrariesReadingRoom = req.querySet.push(allLibrariesReadingRoom) - 1;

    let haveLibrariesReadingRoom = allLibrariesReadingRoom.clone();
    haveLibrariesReadingRoom.where('escola_agregada.sala_leitura = 1');
    req.queryIndex.haveLibrariesReadingRoom = req.querySet.push(haveLibrariesReadingRoom) - 1;

    // Laboratório de informática
    let allInfLab = allSchools.clone();
    allInfLab.where('escola_agregada.func_predio_escolar = 1')
    .where('etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1');
    req.queryIndex.allInfLab = req.querySet.push(allInfLab) - 1;

    let haveInfLab = allInfLab.clone();
    haveInfLab.where('escola_agregada.lab_informatica = 1');
    req.queryIndex.haveInfLab = req.querySet.push(haveInfLab) - 1;

    // Laboratório de ciências
    let allScienceLab = allInfLab.clone();
    req.queryIndex.allScienceLab = req.querySet.push(allScienceLab) - 1;

    let haveScienceLab = allScienceLab.clone();
    haveScienceLab.where('escola_agregada.lab_ciencias = 1 AND (etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.haveScienceLab = req.querySet.push(haveScienceLab) - 1;

    // Parque infantil
    let allKidsPark = allSchools.clone();
    allKidsPark.where('escola_agregada.func_predio_escolar = 1')
    .where('escola_agregada.etapa_ed_infantil_creche = 1 OR escola_agregada.etapa_en_fundamental_anos_iniciais = 1');
    req.queryIndex.allKidsPark = req.querySet.push(allKidsPark) - 1;

    let haveKidsPark = allKidsPark.clone();
    haveKidsPark.where('escola_agregada.parque_infantil = 1');
    req.queryIndex.haveKidsPark = req.querySet.push(haveKidsPark) - 1;

    // Berçário
    let allCribs = allSchools.clone();
    allCribs.where('escola_agregada.func_predio_escolar = 1')
    .where('escola_agregada.reg_infantil_creche = 1 OR escola_agregada.esp_infantil_creche = 1');
    req.queryIndex.allCribs = req.querySet.push(allCribs) - 1;

    let haveCribs = allCribs.clone();
    haveCribs.where('escola_agregada.bercario = 1');
    req.queryIndex.haveCribs = req.querySet.push(haveCribs) - 1;

    // Quadra
    let allSportsCourt = allScienceLab.clone();
    allSportsCourt.where('escola_agregada.localizacao_id = 1 AND (etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.allSportsCourt = req.querySet.push(allSportsCourt) - 1;

    let haveSportsCourt = allSportsCourt.clone();
    haveSportsCourt.where('escola_agregada.quadra_esportes = 1 AND (etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.haveSportsCourt = req.querySet.push(haveSportsCourt) - 1;

    // Quadra coberta
    req.queryIndex.allCoveredSportsCourt = req.queryIndex.allSportsCourt;

    let haveCoveredSportsCourt = allSportsCourt.clone();
    haveCoveredSportsCourt.where('escola_agregada.quadra_esportes_coberta = 1 AND (etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.haveCoveredSportsCourt = req.querySet.push(haveCoveredSportsCourt) - 1;

    // Quadra Descoberta
    let allUncoveredSportsCourt = allSportsCourt.clone();
    allUncoveredSportsCourt.where('escola_agregada.quadra_esportes_coberta = 0 AND (etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.allUncoveredSportsCourt = req.querySet.push(allUncoveredSportsCourt) - 1;

    let haveUncoveredSportsCourt = allUncoveredSportsCourt.clone();
    haveUncoveredSportsCourt.where('escola_agregada.quadra_esportes_descoberta = 1 AND (etapa_en_fundamental_anos_iniciais = 1 OR etapa_en_fundamental_anos_finais = 1 OR etapa_en_medio = 1)');
    req.queryIndex.haveUncoveredSportsCourt = req.querySet.push(haveUncoveredSportsCourt) - 1;

    // Sala de direção
    let allDirectorRoom = allSchools.clone();
    allDirectorRoom.where('escola_agregada.func_predio_escolar = 1 AND escola_agregada.localizacao_id = 1');
    req.queryIndex.allDirectorRoom = req.querySet.push(allDirectorRoom) - 1;

    let haveDirectorRoom = allDirectorRoom.clone();
    haveDirectorRoom.where('escola_agregada.sala_diretoria = 1');
    req.queryIndex.haveDirectorRoom = req.querySet.push(haveDirectorRoom) - 1;

    // Secretaria
    let allSecretary = allSchools.clone();
    allSecretary.where('escola_agregada.func_predio_escolar = 1');
    req.queryIndex.allSecretary = req.querySet.push(allSecretary) - 1;

    let haveSecretary = allSecretary.clone();
    haveSecretary.where('escola_agregada.secretaria = 1');
    req.queryIndex.haveSecretary = req.querySet.push(haveSecretary) - 1;

    // Sala de professores
    req.queryIndex.allTeacherRoom = req.queryIndex.allSecretary;

    let haveTeacherRoom = allSecretary.clone();
    haveTeacherRoom.where('escola_agregada.sala_professor = 1');
    req.queryIndex.haveTeacherRoom = req.querySet.push(haveTeacherRoom) - 1;

    // Cozinha
    req.queryIndex.allKitchen = req.queryIndex.allSecretary;

    let haveKitchen = allSecretary.clone();
    haveKitchen.where('escola_agregada.cozinha = 1');
    req.queryIndex.haveKitchen = req.querySet.push(haveKitchen) - 1;

    // Despensa
    req.queryIndex.allStoreroom = req.queryIndex.allSecretary;

    let haveStoreroom = allSecretary.clone();
    haveStoreroom.where('escola_agregada.despensa = 1');
    req.queryIndex.haveStoreroom = req.querySet.push(haveStoreroom) - 1;

    // Almoxarifado
    req.queryIndex.allWarehouse = req.queryIndex.allSecretary;

    let haveWarehouse = allSecretary.clone();
    haveWarehouse.where('escola_agregada.almoxarifado = 1');
    req.queryIndex.haveWarehouse = req.querySet.push(haveWarehouse) - 1;

    // Internet
    req.queryIndex.allInternet = req.queryIndex.allLibrariesReadingRoom;

    let haveInternet = allLibrariesReadingRoom.clone();
    haveInternet.where('escola_agregada.internet = 1 AND localizacao_id = 2');
    req.queryIndex.haveInternet = req.querySet.push(haveInternet) - 1;

    // Internet banda larga
    req.queryIndex.allBroadbandInternet = req.queryIndex.allLibraries;

    let haveBroadbandInternet = allLibraries.clone();
    haveBroadbandInternet.where('escola_agregada.internet_banda_larga = 1 AND localizacao_id = 1');
    req.queryIndex.haveBroadbandInternet = req.querySet.push(haveBroadbandInternet) - 1;

    // Banheiro dentro do prédio
    req.queryIndex.allInsideBathroom = req.queryIndex.allSecretary;

    let haveInsideBathroom = allSecretary.clone();
    haveInsideBathroom.where('escola_agregada.sanitario_dentro_predio = 1');
    req.queryIndex.haveInsideBathroom = req.querySet.push(haveInsideBathroom) - 1;

    // Banheiro adequado para educação infantil dentro do prédio
    req.queryIndex.allInsideKidsBathroom = req.queryIndex.allKidsPark;

    let haveInsideKidsBathroom = allKidsPark.clone();
    haveInsideKidsBathroom.where('escola_agregada.sanitario_ei = 1 AND (escola_agregada.etapa_ed_infantil_creche = 1 OR escola_agregada.etapa_en_fundamental_anos_iniciais = 1) AND localizacao_id IN (1, 2)');
    req.queryIndex.haveInsideKidsBathroom = req.querySet.push(haveInsideKidsBathroom) - 1;

    // Fornecimento de energia
    req.queryIndex.allEletricEnergy = req.queryIndex.allSecretary;

    let haveEletricEnergy = allSecretary.clone();
    haveEletricEnergy.where('escola_agregada.fornecimento_energia = 1');
    req.queryIndex.haveEletricEnergy = req.querySet.push(haveEletricEnergy) - 1;

    // Abastecimento de água
    req.queryIndex.allWaterSupply = req.queryIndex.allSecretary;

    let haveWaterSupply = allSecretary.clone();
    haveWaterSupply.where('escola_agregada.fornecimento_agua = 1');
    req.queryIndex.haveWaterSupply = req.querySet.push(haveWaterSupply) - 1;

    // Água filtrada
    req.queryIndex.allFilteredWater = req.queryIndex.allSecretary;

    let haveFilteredWater = allSecretary.clone();
    haveFilteredWater.where('escola_agregada.agua_filtrada = 1');
    req.queryIndex.haveFilteredWater = req.querySet.push(haveFilteredWater) - 1;

    // Coleta de esgoto
    req.queryIndex.allSewage = req.queryIndex.allSecretary;

    let haveSewage = allSecretary.clone();
    haveSewage.where('escola_agregada.esgoto_sanitario = 1');
    req.queryIndex.haveSewage = req.querySet.push(haveSewage) - 1;

    // Sala de recursos multifuncionais para Atendimento Educacional Especializado
    req.queryIndex.allMultifunctionRoom = req.queryIndex.allSecretary;

    let haveMultifunctionRoom = allSecretary.clone();
    haveMultifunctionRoom.where('escola_agregada.sala_atendimento_especial = 1');
    req.queryIndex.haveMultifunctionRoom = req.querySet.push(haveMultifunctionRoom) - 1;

    // Banheiros adaptados para pessoas com deficiências
    req.queryIndex.allSpecialBathroom = req.queryIndex.allSecretary;

    let haveSpecialBathroom = allSecretary.clone();
    haveSpecialBathroom.where('escola_agregada.sanitario_pne = 1');
    req.queryIndex.haveSpecialBathroom = req.querySet.push(haveSpecialBathroom) - 1;

    // Dependências adaptada para pessoas com deficiências
    req.queryIndex.allAdaptedBuilding = req.queryIndex.allSecretary;

    let haveAdaptedBuilding = allSecretary.clone();
    haveAdaptedBuilding.where('escola_agregada.dependencias_pne = 1');
    req.queryIndex.haveAdaptedBuilding = req.querySet.push(haveAdaptedBuilding) - 1;


    next();
}, multiQuery, (req, res, next) => {
    // Faz o matching entre os resultados
    let school_place = matchQueries(req.result[req.queryIndex.allSchools], req.result[req.queryIndex.schoolPlace]);
    let libraries = matchQueries(req.result[req.queryIndex.allLibraries], req.result[req.queryIndex.haveLibraries]);
    let libraries_reading_room = matchQueries(req.result[req.queryIndex.allLibrariesReadingRoom], req.result[req.queryIndex.haveLibrariesReadingRoom]);
    let computer_lab = matchQueries(req.result[req.queryIndex.allInfLab], req.result[req.queryIndex.haveInfLab]);
    let science_lab = matchQueries(req.result[req.queryIndex.allScienceLab], req.result[req.queryIndex.haveScienceLab]);
    let kids_park = matchQueries(req.result[req.queryIndex.allKidsPark], req.result[req.queryIndex.haveKidsPark]);
    let nursery = matchQueries(req.result[req.queryIndex.allCribs], req.result[req.queryIndex.haveCribs]);
    let sports_court = matchQueries(req.result[req.queryIndex.allSportsCourt], req.result[req.queryIndex.haveSportsCourt]);
    let covered_sports_court = matchQueries(req.result[req.queryIndex.allCoveredSportsCourt], req.result[req.queryIndex.haveCoveredSportsCourt]);
    let uncovered_sports_court = matchQueries(req.result[req.queryIndex.allUncoveredSportsCourt], req.result[req.queryIndex.haveUncoveredSportsCourt]);
    let director_room = matchQueries(req.result[req.queryIndex.allDirectorRoom], req.result[req.queryIndex.haveDirectorRoom]);
    let secretary = matchQueries(req.result[req.queryIndex.allSecretary], req.result[req.queryIndex.haveSecretary]);
    let teacher_room = matchQueries(req.result[req.queryIndex.allTeacherRoom], req.result[req.queryIndex.haveTeacherRoom]);
    let kitchen = matchQueries(req.result[req.queryIndex.allKitchen], req.result[req.queryIndex.haveKitchen]);
    let storeroom = matchQueries(req.result[req.queryIndex.allStoreroom], req.result[req.queryIndex.haveStoreroom]);
    let warehouse = matchQueries(req.result[req.queryIndex.allWarehouse], req.result[req.queryIndex.haveWarehouse]);
    let internet = matchQueries(req.result[req.queryIndex.allInternet], req.result[req.queryIndex.haveInternet]);
    let broadband_internet = matchQueries(req.result[req.queryIndex.allBroadbandInternet], req.result[req.queryIndex.haveBroadbandInternet]);
    let inside_bathroom = matchQueries(req.result[req.queryIndex.allInsideBathroom], req.result[req.queryIndex.haveInsideBathroom]);
    let inside_kids_bathroom = matchQueries(req.result[req.queryIndex.allInsideKidsBathroom], req.result[req.queryIndex.haveInsideKidsBathroom]);
    let eletric_energy = matchQueries(req.result[req.queryIndex.allEletricEnergy], req.result[req.queryIndex.haveEletricEnergy]);
    let water_supply = matchQueries(req.result[req.queryIndex.allWaterSupply], req.result[req.queryIndex.haveWaterSupply]);
    let filtered_water = matchQueries(req.result[req.queryIndex.allFilteredWater], req.result[req.queryIndex.haveFilteredWater]);
    let sewage_treatment = matchQueries(req.result[req.queryIndex.allSewage], req.result[req.queryIndex.haveSewage]);
    let special_multifunction_room = matchQueries(req.result[req.queryIndex.allMultifunctionRoom], req.result[req.queryIndex.haveMultifunctionRoom]);
    let special_bathroom = matchQueries(req.result[req.queryIndex.allSpecialBathroom], req.result[req.queryIndex.haveSpecialBathroom]);
    let adapted_building = matchQueries(req.result[req.queryIndex.allAdaptedBuilding], req.result[req.queryIndex.haveAdaptedBuilding]);

    req.result = [{
        school_place,
        libraries,
        libraries_reading_room,
        computer_lab,
        science_lab,
        kids_park,
        nursery,
        sports_court,
        covered_sports_court,
        uncovered_sports_court,
        director_room,
        secretary,
        teacher_room,
        kitchen,
        storeroom,
        warehouse,
        internet,
        broadband_internet,
        inside_bathroom,
        inside_kids_bathroom,
        eletric_energy,
        water_supply,
        filtered_water,
        sewage_treatment,
        special_multifunction_room,
        special_bathroom,
        adapted_building
    }];

    next();
}, id2str.multitransform(false), response('infrastructure'));

module.exports = infrastructureApp;
