/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const transportApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const id2str = require(`${libs}/middlewares/id2str`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

transportApp.use(cache('15 day'));

transportApp.get('/year_range', (req, res, next) => {
    req.sql.from('transporte')
    .field('MIN(transporte.ano_censo)', 'start_year')
    .field('MAX(transporte.ano_censo)', 'end_year');
    next();
}, query, response('range'));

transportApp.get('/years', (req, res, next) => {
    req.sql.from('transporte')
    .field('DISTINCT transporte.ano_censo', 'year');
    next();
}, query, response('years'));

transportApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

transportApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 6; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

transportApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

transportApp.get('/rural_location', (req, res, next) => {
    req.result = [
        {id: 1, name: "Urbana"},
        {id: 2, name: "Rural"},
        {id: 3, name: "Rural - Área de assentamento"},
        {id: 4, name: "Rural - Terra indígena"},
        {id: 5, name: "Rural - Área remanescente de quilombos"},
        {id: 6, name: "Rural - Unidade de uso sustentável"}
    ];
    next();
}, response('rural_location'));

transportApp.get('/education_level_mod', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Creche'},
        {id: 2, name: 'Pré-Escola'},
        {id: 4, name: 'Ensino Fundamental - anos iniciais'},
        {id: 5, name: 'Ensino Fundamental - anos finais'},
        {id: 6, name: 'Ensino Médio'},
        {id: 7, name: 'Turmas multiseriadas e multietapas'},
        {id: 8, name: 'EJA - Ensino Fundamental'},
        {id: 9, name: 'EJA - Ensino Médio'},
        {id: 10, name: 'Educação Profissional'}
    ];
    next();
}, response('education_level_mod'));

transportApp.get('/service_type', (req, res, next) => {
    req.result = [
        {id: 0, name: 'Não se aplica'},
        {id: 1, name: 'Classe hospitalar'},
        {id: 2, name: 'Unidade de Atendimento Socioeducativo'},
        {id: 3, name: 'Unidade Prisional'},
        {id: 4, name: 'Atividade Complementar '},
        {id: 5, name: 'Atendimento Educacional Especializado (AEE)'}
    ];
    next();
}, response('service_type'));

transportApp.get('/transportation_manager', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Estadual'},
        {id: 2, name: 'Municipal'},
    ];
    next();
}, response('transportation_manager'));

transportApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'matricula\'');
    next();
}, query, response('source'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'transporte'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'transporte'
    }
}).addValue({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'municipio_id',
        table: 'transporte'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'transporte'
    }
}).addValue({
    name: 'state',
    table: 'transporte',
    tableField: ['estado_nome', 'estado_id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'estado_id',
        table: 'transporte'
    }
}).addValue({
    name: 'rural_location',
    table: 'transporte',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'location',
    table: 'transporte',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name:'adm_dependency',
    table: 'transporte',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'transporte',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'transportation_manager',
    table: 'transporte',
    tableField: 'responsavel_transp',
    resultField: 'transportation_manager_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'responsavel_transp'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'transporte',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
    name: 'service_type',
    table: 'transporte',
    tableField: 'tipo',
    resultField: 'service_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tipo'
    }
}).addValue({
    name: 'min_year',
    table: 'transporte',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'transporte',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'year',
    table: 'transporte',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '=',
        type: 'integer',
        field: 'ano_censo'
    }
});

transportApp.get('/', rqf.parse(), (req, res, next) => {
    req.dims.year = true;
    req.sql
        .field('sum(transporte.total)', 'total')
        .field("'Brasil'", 'name')
        .from('transporte')
        .where('transporte.transporte_id=0')
    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    req.total = req.result;
    req.resetSql();
    next();
}, rqf.parse(), (req, res, next) => {
    req.dims.year = true;
    req.sql
        .field('sum(transporte.total)', 'total')
        .field("'Brasil'", 'name')
        .from('transporte')
        .where('transporte.transporte_id=1')
    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    req.public_total = req.result;
    req.resetSql();
    next();

}, rqf.parse(), (req, res, next) => {
    req.dims.year = true;
    req.sql
        .field('sum(transporte.total)', 'total')
        .field("'Brasil'", 'name')
        .field('transporte.transporte_id', 'id')
        .from('transporte')
        .where('transporte.transporte_id>0')
        .group('transporte.transporte_id')
        .order('transporte.transporte_id')
    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    let transports = req.result;

    let results = [];
    let obj = {};
    for (let i = 1; i < 13; i++) {
        obj[id2str.transport(i)] = [];
    }

    let i = 0
    while (i < transports.length) {
        let result = [];
        let j = 0;
        let transport = transports[i];
        let totalArray = (transport.id  == 1) ? req.total : req.public_total;
        let match;
        obj[id2str.transport(transport.id)] = result;
        while (j < totalArray.length && i < transports.length) {
            transport = transports[i];
            let transportTotal = totalArray[j];

            let currentTransport = {};
            delete transport.id;
            match = true;
            Object.keys(transport).forEach(function(key) {
                currentTransport[key] = transportTotal[key];
                if (key != 'total' && transport[key] != transportTotal[key]) {
                    match = false;
                    return;
                }
            })

            if (match) {
                currentTransport.partial = (match) ? transport.total : 0;
                currentTransport.percentage = (match) ? transport.total/transportTotal.total * 100 : 0;
                result.push(currentTransport);
                i++;
                j++;
            }
            else
                j++;
        }
    }
    results.push(obj)
    req.result = results;

    next();
}, response('transports'));

module.exports = transportApp;
