process.env.NODE_ENV = 'test';

const chai = require('chai');

const dirtyChai = require('dirty-chai');

chai.use(dirtyChai);

const chaiXml = require('chai-xml');

chai.use(chaiXml);

const chaiHttp = require('chai-http');

const assert = chai.assert;

const expect = chai.expect;

const should = chai.should(); // actually call the function

const libs = `${process.cwd()}/libs`;

const server = require(`${libs}/app`);

const id2str = require(`${libs}/middlewares/id2str`);

chai.use(chaiHttp);

describe('id2str middleware', () => {
    let req, res;
    it('should return a function', (done) => {
        expect(id2str.transform).to.be.a.Function;
        done();
    });

    it('should transform a gender id', (done) => {
        expect(id2str.gender(1)).to.deep.equal('Masculino');
        done();
    });

    it('should transform a period id', (done) => {
        expect(id2str.period(1)).to.deep.equal('Matutino');
        done();
    });

    it('should transform a location id', (done) => {
        expect(id2str.location(1)).to.deep.equal('Urbana');
        expect(id2str.location(2)).to.deep.equal('Rural');
        expect(id2str.location(3)).to.deep.equal('Área de assentamento');
        expect(id2str.location(4)).to.deep.equal('Terra indígena');
        expect(id2str.location(5)).to.deep.equal('Área remanescente de quilombos');
        expect(id2str.location(6)).to.deep.equal('Unidade de uso sustentável');
        done();
    });

    it('should transform a school year id', (done) => {
        expect(id2str.schoolYear(11)).to.deep.equal('Creche - Menor de 1 ano');
        done();
    });


    it('should transform a income level id', (done) => {
        expect(id2str.incomeLevel(1)).to.deep.equal('1º quintil – 20% menores');
        expect(id2str.incomeLevel(12)).to.deep.equal('Não classificada');
        done();
    });

    it('should transform a education level basic id', (done) => {
        expect(id2str.educationLevelBasic(7)).to.deep.equal('Não classificada');
        done();
    });

    it('should transform a age range id', (done) => {
        expect(id2str.ageRange(12)).to.deep.equal('Não declarada');
        done();
    });

    it('should transform a booleans variables', (done) => {
        expect(id2str.booleanVariable(1)).to.deep.equal('Sim');
        expect(id2str.booleanVariable(0)).to.deep.equal('Não');
        expect(id2str.booleanVariable(null)).to.deep.equal('Não Declarado');
        done();
    });

    it('should transform a city size id', (done) => {
        expect(id2str.citySize(10)).to.deep.equal('Não classificada');
        done();
    });

    it('should transform a educatoin type id', (done) => {
        expect(id2str.educationType(10)).to.deep.equal('Não definido');
        done();
    });


    it('should transform a gender Pnad id', (done) => {
        expect(id2str.genderPnad(2)).to.deep.equal('Masculino');
        expect(id2str.genderPnad(4)).to.deep.equal('Feminino');
        done();
    });

    it('should transform a adm dependency id', (done) => {
        expect(id2str.admDependency(6)).to.deep.equal('Não classificada');
        done();
    });

    it('should transform a adm dependency priv id', (done) => {
        expect(id2str.admDependencyPriv(7)).to.deep.equal('Não classificada');
        done();
    });

    it('should transform a ethnic group id', (done) => {
        expect(id2str.ethnicGroup(7)).to.deep.equal('Não declarada');
        done();
    });

    it('should transform a ethnic group Pnad id', (done) => {
        expect(id2str.ethnicGroupPnad(7)).to.deep.equal('Sem declaração');
        done();
    });

    it('should transform a full age range id', (done) => {
        expect(id2str.fullAgeRange(1)).to.deep.equal('0-3');
        expect(id2str.fullAgeRange(2)).to.deep.equal('4-5');
        expect(id2str.fullAgeRange(3)).to.deep.equal('6-10');
        expect(id2str.fullAgeRange(4)).to.deep.equal('11-14');
        expect(id2str.fullAgeRange(5)).to.deep.equal('15-17');
        expect(id2str.fullAgeRange(6)).to.deep.equal('18-24');
        expect(id2str.fullAgeRange(7)).to.deep.equal('25-29');
        expect(id2str.fullAgeRange(8)).to.deep.equal('30-40');
        expect(id2str.fullAgeRange(9)).to.deep.equal('41-50');
        expect(id2str.fullAgeRange(10)).to.deep.equal('51-64');
        expect(id2str.fullAgeRange(11)).to.deep.equal('64+');
        expect(id2str.fullAgeRange(12)).to.deep.equal('Não declarada');

        done();
    });

    it('should transform a idhm level id', (done) => {
        expect(id2str.idhmLevel(12)).to.deep.equal('Não classificada');
        done();
    });


    it('should transform a result', (done) => {
        let req = {
            result: [{gender_id: 2, period_id: 3, school_year_id: 11}]
        };
        id2str.transform(false)(req, {},  (error)=> {
            if (error) { throw new Error('Expected not to receive an error'); }
            req.should.have.property('result');
            req.result.should.not.be.undefined;
            req.result.should.be.deep.equal([{gender_id: 2, period_id: 3, school_year_id: 11, gender_name: 'Feminino', period_name: 'Noturno', school_year_name: 'Creche - Menor de 1 ano'}]);
            done();
        });
    });

    it('should transform a result and delete the ids', (done) => {
        let req = {
            result: [{gender_id: 2, period_id: 3, school_year_id: 11}]
        };
        id2str.transform(true)(req, {},  (error)=>{
            if (error) { throw new Error('Expected not to receive an error'); }
            req.should.have.property('result');
            req.result.should.not.be.undefined;
            req.result.should.be.deep.equal([{gender_name: 'Feminino', period_name: 'Noturno', school_year_name: 'Creche - Menor de 1 ano'}]);
            done();
        });
    });
});
