const Sequelize = require("sequelize");
const crypto = require('crypto');
const db = require('../db/postgres.js');
const libs = `${process.cwd()}/libs`;
const Role = require(`${libs}/models/role`);

// set up a sequelize model 
var User = db.define("User",{
    id:{
        type: Sequelize.STRING,
        allowNull:false,
        unique: true,
        primaryKey: true
    },
    email: {
        type: Sequelize.STRING,
        allowNull: false,
        unique: true,
        validate: {
            notNull: { msg: "O campo Email é obrigatório." },
            notEmpty: { msg: "O campo Email é obrigatório." }
        }
    },
    password:{
        type: Sequelize.VIRTUAL
    },
    hashed_password:{
        type: Sequelize.STRING,
        allowNull: false,
    },
    salt: {
        type: Sequelize.STRING,
        allowNull: false,
    },
    name:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Nome é obrigatório." },
            notEmpty: { msg: "O campo Nome é obrigatório." }
        }
    },
    nickname:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Apelido é obrigatório." },
            notEmpty: { msg: "O campo Apelido é obrigatório." }
        }
    },
    cpf:{
        type: Sequelize.STRING,
        allowNull: false,
        unique: true,
        validate: {
            notNull: { msg: "O campo CPF é obrigatório." },
            notEmpty: { msg: "O campo CPF é obrigatório." }
        }
    },
    cep:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo CEP é obrigatório." },
            notEmpty: { msg: "O campo CEP é obrigatório." }
        }
    },
    schooling:{
        type: Sequelize.STRING,
        allowNull: false, 
        validate: {
            notNull: { msg: "O campo Formação é obrigatório." },
            notEmpty: { msg: "O campo Formação é obrigatório." }
        }
    },
    course:{
        type: Sequelize.STRING,
    },
    complement:{
        type: Sequelize.STRING
    },
    address:{
        type: Sequelize.STRING,
    },
    phone:{
        type: Sequelize.STRING,
    },
    segment:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Segmento é obrigatório." },
            notEmpty: { msg: "O campo Segmento é obrigatório." }
        }
    },
    role:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Função é obrigatório." },
            notEmpty: { msg: "O campo Função é obrigatório." }
        }
    },
    institution_name:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Instituição em que trabalha ou estuda é obrigatório." },
            notEmpty: { msg: "O campo Instituição em que trabalha ou estuda é obrigatório." }
        }
    },
    state:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Estado é obrigatório." },
            notEmpty: { msg: "O campo Estado é obrigatório." }
        }
    },
    city:{
        type: Sequelize.STRING,
        allowNull: false,
        validate: {
            notNull: { msg: "O campo Cidade é obrigatório." },
            notEmpty: { msg: "O campo Cidade é obrigatório." }
        }
    },
    receive_email:{
        type: Sequelize.BOOLEAN,
    },
    created_at:{
        type: Sequelize.DATE,
        defaultValue: Date.now
    },
    origin:{
        type: Sequelize.ENUM("LDE", "SimCAQ", "MAPFOR"),
        allowNull:false,
        validate: {
            notNull: { msg: "O campo origem é obrigatória e aceita apenas os valores 'LDE', 'SimCAQ' e 'MAPFOR'."},
            notEmpty: { msg: "O campo origem é obrigatória e aceita apenas os valores 'LDE', 'SimCAQ' e 'MAPFOR'."}
        }
    },
    verified:{
        type: Sequelize.BOOLEAN,
        defaultValue:false
    },
    citesegment:{
        type: Sequelize.STRING,
    },
    citerole:{
        type: Sequelize.STRING
    },
    admin:{
        type: Sequelize.BOOLEAN,
        defaultValue:false
    },
    role_id:{
        type: Sequelize.NUMBER,
        allowNull:false,
        defaultValue: 0
    }
},
    {timestamps: false}
);

User.hasOne(Role, {foreignKey: 'id'});

User.generateSalt = function() {
    return crypto.randomBytes(128).toString('hex');
}

User.encryptPassword = function(password, salt) {
    return crypto.pbkdf2Sync(password+'', salt, 10000, 512, 'sha512').toString('hex');
}

User.generateObjectId = function(){
    var timestamp = (new Date().getTime() / 1000 | 0).toString(16);
    return timestamp + 'xxxxxxxxxxxxxxxx'.replace(/[x]/g, function() {
        return (Math.random() * 16 | 0).toString(16);
    }).toLowerCase();
}

const setSaltAndPassword = user => {
    if (user.changed('password')) {
        user.salt = User.generateSalt()
        user.hashed_password = User.encryptPassword(user.password, user.salt).toString('hex');
   }
}

const setObjectId = user => {
   user.id = User.generateObjectId()
};

User.beforeCreate(setSaltAndPassword)
User.beforeCreate(setObjectId)
User.beforeUpdate(setSaltAndPassword)

User.prototype.checkPassword = function(user, enteredPassword) {
    return User.encryptPassword(enteredPassword, user.salt) === user.hashed_password
}

module.exports = User; 
