/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const rateSchoolNewApp = express.Router();

const libs = `${process.cwd()}/libs`;

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

rateSchoolNewApp.use(cache('15 day'));

rateSchoolNewApp.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    .where('pnad_novo.ano_ref> 2018')
    next();
}, query, response('years'));

rateSchoolNewApp.get('/region', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.regionCode(i)
        });
    }

    next();
}, response('region'));

rateSchoolNewApp.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

rateSchoolNewApp.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

rateSchoolNewApp.get('/gender', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

rateSchoolNewApp.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

rateSchoolNewApp.get('/state', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.stateName(i) !== 'Não declarada') {
            req.result.push({
                id: i, name: id2str.stateName(i)
            });
        }
    }
    req.result.push({id: 99, name: id2str.stateName(99)});

    next();
}, response('state'));

rateSchoolNewApp.get('/new_pnad_ethnic_group', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.ethnicGroupNewPnad(i)
        });
    }
    req.result.push({id: 9, name: id2str.ethnicGroupNewPnad(9)});
    next();
}, response('new_pnad_ethnic_group'));


rateSchoolNewApp.get('/income_range', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.incomeRange(i)
        });
    }
    req.result.push({id: 9, name: id2str.incomeRange(9)});
    next();
}, response('income_range'));

rateSchoolNewApp.get('/age_range', (req, res, next) => {
    req.result = [
        {id: 1, name: '0 a 3 anos'},
        {id: 2, name: '4 a 5 anos'},
        {id: 3, name: '6 a 10 anos'},
        {id: 4, name: '11 a 14 anos'},
        {id: 5, name: '15 a 17 anos'},
        {id: 6, name: '18 a 24 anos'}
    ];
    next();
}, response('age_range'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'state_not',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '<>',
        type: 'integer',
        field: 'cod_uf',
        table: 'pnad_novo'
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'age_range',
    table: 'pnad_novo',
    tableField: 'faixa_etaria',
    resultField: 'age_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux'
    }
});

/* 
    The queries are matched and the total is calculated.
    This function is necessary to match the results of the
    2 queries with the right pair. In some queries, the result
    doesn't exist in one of the 2, so, we need to run this algorithm
    to match them right.
*/

function matchQueries(queryPartial, queryTotal) {
    let match = [];
    queryTotal.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryPartial.length; ++i) {
            let partial = queryPartial[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(partial[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = partial;
                break;
            }
        }

        if(objMatch) {
            newObj.denominator = result.total;
            newObj.partial = objMatch.total;
            newObj.total = (objMatch.total / result.total) * 100;
            match.push(newObj);
        }
    });

    return match;
}

rateSchoolNewApp.get('/', rqf.parse(), rqf.build(),  (req, res, next) => {
    
    if ("age_range" in req.filter || "age_range" in req.dims) {
        // As we will need to do two requests, they'll be stored in a list
        req.querySet = []
        
        // Create an object that will store the first request (the sum of all people that attend school)
        // and are below a certain age (in this case, 24 yeas)
        let attends_school = req.sql.clone();
        attends_school.from('pnad_novo')
        .field('round(sum(pnad_novo.peso_domicilio_pessoas_com_cal), 0)', 'total')
        .field('pnad_novo.faixa_etaria')
        .field('pnad_novo.ano_ref', 'year')
        .where('pnad_novo.ano_ref >= 2019 AND frequenta_escola = 1')
        .where('pnad_novo.faixa_etaria < 7')
        .group('pnad_novo.ano_ref')
        .group('pnad_novo.faixa_etaria')
        .order('pnad_novo.ano_ref')
        .order('pnad_novo.faixa_etaria')
        req.querySet.push(attends_school);

        // Then, the second object is created and stores the sum of all people below a certain age (24 years)
        let full_population = req.sql.clone();
        full_population.from('pnad_novo')
        .field('round(sum(pnad_novo.peso_domicilio_pessoas_com_cal), 0)', 'total')
        .field('pnad_novo.faixa_etaria')
        .field('pnad_novo.ano_ref', 'year')
        .where('pnad_novo.ano_ref >= 2019')
        .where('pnad_novo.faixa_etaria < 7')
        .group('pnad_novo.ano_ref')
        .group('pnad_novo.faixa_etaria')
        .order('pnad_novo.ano_ref')
        .order('pnad_novo.faixa_etaria')
        req.querySet.push(full_population);
    }

    next();
}, multiQuery, (req, res, next) => {
    if ("age_range" in req.filter || "age_range" in req.dims) {
        // The multiple requests are made. Then we need to calculate the percetange. So the function
        // below is used
        let newObj = matchQueries(req.result[0], req.result[1]);
        req.result = newObj;
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }
    next();
}, id2str.transform(false), response('rateSchoolNew'));

module.exports = rateSchoolNewApp;
