/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const liquidEnrollmentRatioApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

liquidEnrollmentRatioApp.use(cache('15 day'));

// Complete range of the enrollments dataset.
// Returns a tuple of start and ending years of the complete enrollments dataset.
liquidEnrollmentRatioApp.get('/year_range', (req, res, next) => {
    req.sql.from('pnad')
    .field('DISTINCT pnad.ano_censo', 'year');
    next();
}, query, (req, res, next) => {
    req.oldResult = req.result;

    req.sql = squel.select();

    req.sql.from('matricula')
    .field('DISTINCT matricula.ano_censo', 'year');
    next();
}, query, (req, res, next) => {
    let distinct_years = [];
    let new_result = [];
    for (let i = 0; i < req.oldResult.length; i++) {
        for (let j = 0; j < req.result.length; j++) {
            if(req.oldResult[i].year == req.result[j].year) {
                distinct_years.push(req.oldResult[i]);
            }
        }
    }
    new_result.push({start_year: distinct_years[distinct_years.length -1].year, end_year: distinct_years[0].year});
    req.result = new_result;
    next();
}, response('range'));

liquidEnrollmentRatioApp.get('/years', (req, res, next) => {
    req.sql.from('pnad')
    .field('DISTINCT pnad.ano_censo', 'year');
    next();
}, query, (req, res, next) => {
    req.oldResult = req.result;

    req.sql = squel.select();

    req.sql.from('matricula')
    .field('DISTINCT matricula.ano_censo', 'year');
    next();
}, query, (req, res, next) => {
    let new_result = []
    for (let i = 0; i < req.oldResult.length; i++) {
        for (let j = 0; j < req.result.length; j++) {
            if(req.oldResult[i].year == req.result[j].year) {
                new_result.push(req.oldResult[i]);
            }
        }
    }
    req.result = new_result;
    next();
}, response('years'));

liquidEnrollmentRatioApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'pnad\'');
    next();
}, query, response('source'));

liquidEnrollmentRatioApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Creche'},
        {id: 2, name: 'Pré-Escola'},
        {id: 3, name: 'Ensino Fundamental - anos iniciais'},
        {id: 4, name: 'Ensino Fundamental - anos finais'},
        {id: 5, name: 'Ensino Médio'}
    ];
    next();
}, response('education_level_short'));

liquidEnrollmentRatioApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));

liquidEnrollmentRatioApp.get('/ethnic_group', (req, res, next) => {
    req.result = [
        {id: 0, name: 'Sem declaração'},
        {id: 1, name: 'Branca'},
        {id: 2, name: 'Preta'},
        {id: 3, name: 'Parda'},
        {id: 4, name: 'Amarela'},
        {id: 5, name: 'Indígena'}
    ];
    next();
}, response('ethnic_group'));

liquidEnrollmentRatioApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: '@'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: '@'
    }
}).addValue({
    name: 'ethnic_group',
    table: '@',
    tableField: 'cor_raca_id',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca_id'
    }
}).addValue({
    name: 'min_year',
    table: '@',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: '@',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: '@',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: '@',
        field: 'ano_censo'
    }
}).addValue({
    name: 'gender',
    table: '@',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'location',
    table: '@',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'education_level_short',
    table: 'matricula',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
});

function matchQueries(queryTotal, queryPartial) {
    let match = [];
    queryPartial.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        // console.log('NEW OBJ');
        // console.log(newObj);
        // remove total
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        // remove education_level_short_id
        index = keys.indexOf('education_level_short_id');
        if(index > -1) keys.splice(index, 1);
        // remove education_level_short_name
        index = keys.indexOf('education_level_short_name');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryTotal.length; ++i) {
            let total = queryTotal[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(total[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = total;
                break;
            }
        }

        if(objMatch) {
            // console.log('MATCH!!!!');
            // console.log(objMatch);
            newObj.total = (result.total / objMatch.total) * 100;
            newObj.partial = result.total;
            newObj.denominator = objMatch.total
            match.push(newObj);
        }
    });
    // console.log('TAMANHOS');
    // console.log(queryTotal.length);
    // console.log(queryPartial.length);
    // console.log(match.length);
    return match;
}

function ConvertEnrollment(result) {
    if (result == 1) {
        return '(matricula.faixa_etaria_31_03 = 1 AND matricula.etapa_resumida = 1)'
    } else if (result == 2) {
        return '(matricula.faixa_etaria_31_03 = 2 AND matricula.etapa_resumida = 2)'
    } else if (result == 4) {
        return '(matricula.faixa_etaria_31_03 = 3 AND matricula.etapa_resumida = 3)'
    } else if (result == 5) {
        return '(matricula.faixa_etaria_31_03 = 4 AND matricula.etapa_resumida = 4)'
    } else if (result == 6) {
        return '(matricula.faixa_etaria_31_03 = 5 AND matricula.etapa_resumida = 5)'
    }
}


function convertPnad(result) {
    if (result == 1) {
        return 'pnad.faixa_etaria_31_03 = 1'
    } else if (result == 2) {
        return 'pnad.faixa_etaria_31_03 = 2'
    } else if (result == 4) {
        return 'pnad.faixa_etaria_31_03 = 3'
    } else if (result == 5) {
        return 'pnad.faixa_etaria_31_03 = 4'
    } else if (result == 6) {
        return 'pnad.faixa_etaria_31_03 = 5'
    }
}

liquidEnrollmentRatioApp.get('/', rqf.parse(),(req, res, next) => {
    req.numerator = {};
    req.denominator = {};
    let liquidEnrollmentRatioApp = {};

    req.sql.from('matricula')
    .field('count(*)', 'total')
    .field('matricula.ano_censo', 'year')
    .group('matricula.ano_censo')
    .order('matricula.ano_censo')
    .where('matricula.tipo <= 3')

    if ("education_level_short" in req.filter) {

        if (Array.isArray(req.filter.education_level_short)) {
            var stringQuery = '';
            for(let i = 0; i < req.filter.education_level_short.length - 1; i++)  {
                stringQuery = stringQuery + ConvertEnrollment(req.filter.education_level_short[i]) + ' OR ';
            }

            stringQuery = stringQuery + ConvertEnrollment(req.filter.education_level_short[req.filter.education_level_short.length - 1]);
            delete req.filter.education_level_short;
            req.sql.where(stringQuery);
            req.sql.field('matricula.faixa_etaria_31_03', 'age_range')
            req.sql.group('matricula.faixa_etaria_31_03', 'age_range');
        }

    } else if ( "education_level_short" in req.dims ) {

        req.sql.field('matricula.faixa_etaria_31_03', 'age_range')
        req.sql.where('(matricula.etapa_resumida = 1  AND matricula.faixa_etaria_31_03 = 1) OR (matricula.etapa_resumida = 2  AND matricula.faixa_etaria_31_03 = 2) OR (matricula.etapa_resumida = 3  AND matricula.faixa_etaria_31_03 = 3) OR (matricula.etapa_resumida = 4  AND matricula.faixa_etaria_31_03 = 4) OR (matricula.etapa_resumida = 5  AND matricula.faixa_etaria_31_03 = 5)');
        req.sql.group('matricula.faixa_etaria_31_03', 'age_range');

    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }

    next();
}, rqf.build(), query, id2str.transform(),  (req, res, next) => {
    req.numerator = req.result;
    req.resetSql();
    req.sql.field('sum(peso)', 'total')
    .field('pnad.ano_censo','year')
    .from('pnad')
    .group('pnad.ano_censo')
    .order('pnad.ano_censo')

    //remove education_level_short how filter and add faixa_etaria_31_03 in filter

    if ("education_level_short" in req.filter) {

        if (Array.isArray(req.filter.education_level_short)) {
            var stringQuery = '';
            for(let i = 0; i < req.filter.education_level_short.length - 1; i++)  {
                stringQuery = stringQuery + convertPnad(req.filter.education_level_short[i]) + ' OR ';
            }
            stringQuery = stringQuery + convertPnad(req.filter.education_level_short[req.filter.education_level_short.length - 1]);
            req.sql.where(stringQuery);
        }
        req.sql.field('pnad.faixa_etaria_31_03', 'age_range')
        req.sql.group('pnad.faixa_etaria_31_03', 'age_range');

    } else if ( "education_level_short" in req.dims ) {

        req.sql.field('pnad.faixa_etaria_31_03','age_range')
        req.sql.where('pnad.faixa_etaria_31_03 = 1 OR pnad.faixa_etaria_31_03 = 2 OR pnad.faixa_etaria_31_03 = 3 OR pnad.faixa_etaria_31_03 = 4 OR pnad.faixa_etaria_31_03 = 5')
        req.sql.group('pnad.faixa_etaria_31_03', 'age_range');
    } else {
        res.status(400);
        next({
            status: 400,
            message: 'Wrong/No filter specified'
        });
    }

    next();
}, rqf.parse(), (req, res, next) => {
    if ("education_level_short" in req.filter) {
        delete req.filter.education_level_short;
    }
    if ("education_level_short" in req.dims) {
        delete req.dims.education_level_short;
    }
    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    req.denominator = req.result;

    //division to generate req.result final
    req.result = []
    let liquidEnrollment = matchQueries(req.denominator, req.numerator);
    req.result = liquidEnrollment;
    next();
}, response('liquidEnrollmentRatio'));

module.exports = liquidEnrollmentRatioApp;
