/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const teacherApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]}  }).middleware;

let rqf = new ReqQueryFields();

teacherApp.use(cache('15 day'));

// Returns a tuple of start and ending years of the complete enrollments dataset.
teacherApp.get('/year_range', (req, res, next) => {
    req.sql.from('docente')
    .field('MIN(docente.ano_censo)', 'start_year')
    .field('MAX(docente.ano_censo)', 'end_year');
    next();
}, query, response('range'));

teacherApp.get('/years', (req, res, next) => {
    req.sql.from('docente').
    field('DISTINCT docente.ano_censo', 'year');
    next();
}, query, response('years'));

teacherApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'docente\'');
    next();
}, query, response('source'));

teacherApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

teacherApp.get('/diff_location', (req, res, next) => {
    req.result = [
        {id: 0, name: "A escola não está em localidade diferenciada"},
        {id: 1, name: "Área de assentamento"},
        {id: 2, name: "Terra indígena"},
        {id: 3, name: "Terra remanescente de quilombos"},
    ];
    next();
}, response('diff_location'));

teacherApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 8; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

teacherApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

teacherApp.get('/education_level_mod', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 12; ++i) {   
        req.result.push({
            id: i,
            name: id2str.educationLevelMod(i)
        });
    }
    req.result.push({
        id: 99,
        name: id2str.educationLevelMod(99)
    });
    next();
}, response('education_level_mod'));

teacherApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Creche'},
        {id: 2, name: 'Pré-Escola'},
        {id: 3, name: 'Ensino Fundamental - anos iniciais'},
        {id: 4, name: 'Ensino Fundamental - anos finais'},
        {id: 5, name: 'Ensino Médio'},
        {id: 6, name: 'EJA'},
        {id: 7, name: 'EE exclusiva'}
    ];
    next();
}, response('education_level_short'));

teacherApp.get('/education_type', (req, res, next) => {
    req.sql.from('docente')
    .field('DISTINCT nivel_tipo_formacao', 'id')
    .order('id');
    next();
}, query, (req, res, next) => {
    req.result.forEach((result) => {
        result.name = id2str.educationType(result.id);
    });
    next();
}, response('education_type'));

teacherApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));


teacherApp.get('/contract_type', (req, res, next) => {
    req.result = [{
        id: "null",
        contractType: id2str.contractType("null")
    }];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.contractType(i)
        });
    }
    next();
}, response('contract_type'));

teacherApp.get('/ethnic_group', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroup(i)
        });
    }
    next();
}, response('ethnic_group'));

teacherApp.get('/initial_training', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.initialTraining(i)
        });
  }
  next();
}, response('initial_training'));

teacherApp.get('/pos_training', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.posTraining(i)
        });
  }
  next();
}, response('pos_training'));

teacherApp.get('/licentiate_degree', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 3; ++i) {
        req.result.push({
            id: i,
            name: id2str.licentiateDegree(i)
        });
  }
  next();
}, response('licentiate_degree'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'adm_dependency',
    table: 'docente',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'docente',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'contract_type',
    table: 'docente',
    tableField: 'tipo_contratacao',
    resultField: 'contract_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tipo_contratacao'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'docente',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
    name: 'education_level_short',
    table: 'docente',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
}).addValue({
    name: 'education_type',
    table: 'docente',
    tableField: 'nivel_tipo_formacao',
    resultField: 'education_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_tipo_formacao'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_regiao_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'mesoregion',
    table: 'municipio',
    tableField: ['nome_mesorregiao', 'mesorregiao_id'],
    resultField: ['mesoregion_name', 'mesoregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'microregion',
    table: 'municipio',
    tableField: ['nome_microrregiao', 'microrregiao_id'],
    resultField: ['microregion_name', 'microregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_estado_id',
        foreignTable: 'docente'
    }
}).addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'escola_municipio_id',
        foreignTable: 'docente'
    }
}, 'filter').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'dims').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: 'nome_escola',
    resultField: 'school_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'docente'
    }
}, 'filter').addValue({
    name: 'location',
    table: 'docente',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'diff_location',
    table: 'docente',
    tableField: 'localizacao_diferenciada_par',
    resultField: 'diff_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_diferenciada_par'
    }
}).addValue({
    name: 'rural_location',
    table: 'docente',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'min_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'docente',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'gender',
    table: 'docente',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'ethnic_group',
    table: 'docente',
    tableField: 'cor_raca',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'initial_training',
    table: 'docente',
    tableField: 'formacao_inicial_docente',
    resultField: 'initial_training_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'formacao_inicial_docente'
    }
}).addValue({
    name: 'pos_training',
    table: 'docente',
    tableField: 'formacao_pos_docente',
    resultField: 'pos_training_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'formacao_pos_docente'
    }
}).addValue({
    name: 'licentiate_degree',
    table: 'docente',
    tableField: 'formacao_licenciatura_docente',
    resultField: 'licentiate_degree_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'formacao_licenciatura_docente'
    }
});

const sortYearPtid = (a, b) => {
    if (a.year < b.year)
        return -1
    if (a.year > b.year)
        return 1
    
    if (a.pos_training_id < b.pos_training_id)
        return -1
    if (a.pos_training_id > b.pos_training_id)
        return 1
    return 0
}

teacherApp.get('/', rqf.parse(), (req, res, next) => {
    req.sql.field('COUNT(DISTINCT docente.id_docente)', 'total')
    .field("'Brasil'", 'name')
    .field('docente.ano_censo', 'year')
    .from('docente')
    .join('turma', null, 'docente.turma_id=turma.id AND docente.ano_censo=turma.ano_censo')
    .group('docente.ano_censo')
    .order('docente.ano_censo')
    .where('(docente.tipo_docente = 1 OR docente.tipo_docente = 5) AND \
        ((docente.tipo_turma_id >= 0 AND docente.tipo_turma_id <= 3 AND docente.tipo_turma_atendimento_id is NULL) \
        OR ((docente.tipo_turma_atendimento_id = 1 OR docente.tipo_turma_atendimento_id = 2) AND docente.tipo_turma_id is NULL)) \
        AND (docente.ano_censo <> 2009 or (docente.escola_estado_id <> 42 and docente.escola_estado_id <> 43) )');     // não devemos trazer SC em 2009.

    // if("education_level_mod" in req.dims) {
    //     req.hadEducationLevelMod = true;
    //     req.sql.where('docente.etapas_mod_ensino_segmento_id < 11');
    // }

    next();
}, rqf.build(), query, addMissing(rqf), (req, res, next) => {
    // Função criada para preencher valores faltantes no gráfico do indicador de 
    // formação em pós graduação do MapFOR.
    if (req.dims.pos_training){
        var year = req.result[0].year;
        var posTrainingIds = req.result.map(obj => {
            if (year == obj.year) 
                return obj.pos_training_id
        }).filter(num => num !== undefined)
        
        var missingValues = [];
        for(let i = 1; i <= 4; ++i) {
            if ( !posTrainingIds.includes(i) ){
                missingValues.push(i);
            }
        }
        
        for (let curYear = 2012; curYear <= 2020; ++curYear){
            for (let ptId of missingValues){
                req.result.push({
                    total:0,
                    name:"Brasil",
                    year:curYear,
                    pos_training_id:ptId,
                    pos_training_name:id2str.posTraining(ptId)
                })
            }
        }
        req.result.sort(sortYearPtid)
        
    }
    next();
}, id2str.transform(), response('teacher'));

module.exports = teacherApp;
