/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const universityEnrollmentApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

universityEnrollmentApp.get('/years', (req, res, next) => {
    req.sql.from('aluno_ens_superior')
    .field('DISTINCT aluno_ens_superior.ano_censo', 'year')
		.where('aluno_ens_superior.ano_censo > 2010');
    next();
}, query, response('years'));

universityEnrollmentApp.get('/year_range', (req, res, next) => {
    req.sql.from('aluno_ens_superior')
    .field('MIN(aluno_ens_superior.ano_censo)', 'start_year')
    .field('MAX(aluno_ens_superior.ano_censo)', 'end_year');
    next();
}, query, response('range'));

universityEnrollmentApp.get('/academic_organization', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 5; ++i) {
        req.result.push({
            id: i,
            name: id2str.academicOrganization(i)
        });
    };
    next();
}, response('academic_organization'));

universityEnrollmentApp.get('/upper_adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 7; ++i) {
        req.result.push({
            id: i,
            name: id2str.upperAdmDependency(i)
        });
    };
    next();
}, response('upper_adm_dependency'));

universityEnrollmentApp.get('/ocde_geral', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <= 8; ++i) {
        req.result.push({
            id: i,
            name: id2str.ocdeGeral(i)
        });
    };
    next();
}, response('ocde_geral'));


universityEnrollmentApp.get('/finish', (req, res, next) => {
    req.result = []
    for (let i = 0; i <= 1; ++i){
        req.result.push({
            id: i,
            name: id2str.finishUniversity(i)
        })
    }
    next();

}, response('finish'));

universityEnrollmentApp.get('/ocde_specific', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 1; i <= 86; ++i) {
        let obj = {
            id: i,
            name: id2str.ocdeSpecific(i)
        };
        if (obj.name !== id2str.ocdeSpecific(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.ocdeSpecific(defaultCase)
    });
    next();
}, response('ocde_specific'));

universityEnrollmentApp.get('/ocde_detailed', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 142; i <= 863; ++i) {
        let obj = {
            id: i,
            name: id2str.ocdeDetailed(i)
        };
        if (obj.name !== id2str.ocdeDetailed(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.ocdeDetailed(defaultCase)
    });
    next();
}, response('ocde_detailed'));

universityEnrollmentApp.get('/cine_geral', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <= 10; ++i) {
        req.result.push({
            id: i,
            name: id2str.cineGeral(i)
        });
    };
    next();
}, response('cine_geral'));

universityEnrollmentApp.get('/cine_specific', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 1; i <= 104; ++i) {
        let obj = {
            id: i,
            name: id2str.cineSpecific(i)
        };
        if (obj.name !== id2str.cineSpecific(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.cineSpecific(defaultCase)
    });
    next();
}, response('cine_specific'));

universityEnrollmentApp.get('/cine_detailed', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 11; i <= 1041; ++i) {
        let obj = {
            id: i,
            name: id2str.cineDetailed(i)
        };
        if (obj.name !== id2str.cineDetailed(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.cineDetailed(defaultCase)
    });
    next();
}, response('cine_detailed'));

universityEnrollmentApp.get('/upper_turn', (req, res, next) => {
    req.result = [{
        id: "null",
        name: id2str.upperTurn("null")
    }];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.upperTurn(i)
        });
    };
    next();
}, response('upper_turn'));

universityEnrollmentApp.get('/student_deficiency', (req, res, next) => {
    req.result = [{
        id: 9,
        name: id2str.studentDeficiency(9)
    }];
    for(let i = 0; i <= 1; ++i) {
        req.result.push({
            id: i,
            name: id2str.studentDeficiency(i)
        });
    };
    next();
}, response('student_deficiency'));

universityEnrollmentApp.get('/ethnic_group_ies', (req, res, next) => {
    req.result = [{
        id: 9,
        name: id2str.ethnicGroupIES(9)
    }];
    for(let i = 0; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroupIES(i)
        });
    }
    next();
}, response('ethnic_group_ies'));

universityEnrollmentApp.get('/school_type', (req, res, next) => {
    req.result = [{
        id: 9,
        name: id2str.schoolType(9)
    }];
    for(let i = 1; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.schoolType(i)
        });
    };
    next();
}, response('school_type'));

universityEnrollmentApp.get('/university', (req, res, next) => {
    req.sql.from('aluno_ens_superior')
    .field('DISTINCT aluno_ens_superior.nome_ies', 'nome')
    .field('aluno_ens_superior.cod_ies', 'cod')
    next();
}, query, response('university'));

universityEnrollmentApp.get('/academic_level', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.academicLevel(i)
        });
    };
    next();
}, response('academic_level'));

universityEnrollmentApp.get('/gender_ies', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.genderIES(i)
        });
    };
    next();
}, response('gender_ies'));

universityEnrollmentApp.get('/upper_education_mod', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 3; ++i) {
        req.result.push({
            id: i,
            name: id2str.upperEducationMod(i)
        });
    };
    next();
}, response('upper_education_mod'));

universityEnrollmentApp.get('/age_student_code', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 6; ++i) {
        req.result.push({
            id: i,
            name: id2str.ageStudentCode(i)
        });
    };
    next();
}, response('age_student_code'));


universityEnrollmentApp.get('/enter_situation/student_enter_situation', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.enterSituation(i)
        });
    };
    next();
}, response('student_enter_situation'));

universityEnrollmentApp.get('/enrollment_situation/student_enter_situation', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.enterSituation(i)
        });
    };
    next();
}, response('student_enter_situation'));


universityEnrollmentApp.get('/enter_situation/student_enrollment_situation', function (req, res, next) {
    req.result = [];
    for (var i = 1; i <= 3; ++i) {
        req.result.push({
            id: i,
            name: id2str.enrollmentSituation(i)
        });
    };
    next();
}, response('student_enrollment_situation'));

universityEnrollmentApp.get('/university', (req, res, next) => {
    req.sql.from('aluno_ens_superior')
    .field('DISTINCT aluno_ens_superior.nome_ies', 'nome')
    .field('aluno_ens_superior.cod_ies', 'cod')
    next();
}, query, response('university'));

universityEnrollmentApp.get('/localoffer', (req, res, next) => {
    req.sql.from('localoferta_ens_superior', 'l')
    .field('DISTINCT l.nome', 'localoffer_name')
    .field('l.cod_local_oferta', 'localoffer_id');
    next();
}, query, response('localoffer'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'localoferta_cod_municipio',
        table: 'localoferta_ens_superior_matricula'
    },
    join: {
        primary: 'id',
        foreign: 'localoferta_cod_municipio',
        foreignTable: 'localoferta_ens_superior_matricula'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'localoferta_cod_uf',
        table: 'localoferta_ens_superior_matricula'
    },
    join: {
        primary: 'id',
        foreign: 'localoferta_cod_uf',
        foreignTable: 'localoferta_ens_superior_matricula'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'localoferta_cod_regiao',
        table: 'localoferta_ens_superior_matricula'
    },
    join: {
        primary: 'id',
        foreign: 'localoferta_cod_regiao',
        foreignTable: 'localoferta_ens_superior_matricula'
    }
}).addValue({
    name: 'localoffer',
    table: 'localoferta_ens_superior_matricula',
    tableField: ['cod_local_oferta', 'localoferta_nome'],
    resultField: ['localoffer_id', 'localoffer_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_local_oferta'
    }
}).addValue({
    name: 'campi',
    table: 'localoferta_ens_superior_matricula',
    tableField: ['cod_local_oferta', 'localoferta_nome'],
    resultField: ['campi_id', 'campi_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_local_oferta'
    }
}).addValue({
    name: 'university',
    table: '@',
    tableField: ['cod_ies', 'nome_ies'],
    resultField: ['university_id', 'university_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_ies'
    }
}).addValue({
    name: 'universityLocalOffer',
    table: '@',
    tableField: ['cod_ies', 'nome_ies'],
    resultField: ['university_id', 'university_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_ies'
    }
}).addValue({
    name:'upper_adm_dependency',
    table: '@',
    tableField: 'par_categoria_administrativa',
    resultField: 'upper_adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_categoria_administrativa'
    }
}).addValue({
    name:'academic_organization',
    table: '@',
    tableField: 'cod_organizacao_academica',
    resultField: 'academic_organization_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_organizacao_academica'
    }
}).addValue({
    name:'ocde_specific',
    table: '@',
    tableField: ['par_cod_ocde_area_especifica'],
    resultField: ['ocde_specific_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_cod_ocde_area_especifica'
    }
}).addValue({
    name:'ocde_geral',
    table: '@',
    tableField: ['par_cod_ocde_area_geral'],
    resultField: ['ocde_geral_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_cod_ocde_area_geral'
    }
}).addValue({
    name:'ocde_detailed',
    table: '@',
    tableField: ['par_cod_ocde_area_detalhada'],
    resultField: ['ocde_detailed_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_cod_ocde_area_detalhada'
    }
}).addValue({
    name:'cine_specific',
    table: '@',
    tableField: ['cod_cine_area_especifica', 'nome_cine_area_especifica'],
    resultField: ['cine_specific_id', 'cine_specific_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_especifica'
    }
}).addValue({
    name:'cine_geral',
    table: '@',
    tableField: ['cod_cine_area_geral', 'nome_cine_area_geral'],
    resultField: ['cine_geral_id', 'cine_geral_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_geral'
    }
}).addValue({
    name:'cine_detailed',
    table: '@',
    tableField: ['cod_cine_area_detalhada', 'nome_cine_area_detalhada'],
    resultField: ['cine_detailed_id', 'cine_detailed_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_detalhada'
    }
}).addValue({
    name:'academic_level',
    table: '@',
    tableField: 'cod_grau_academico',
    resultField: 'academic_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_grau_academico'
    }
}).addValue({
    name:'upper_education_mod',
    table: '@',
    tableField: 'cod_modalidade_ensino',
    resultField: 'upper_education_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_modalidade_ensino'
    }
}).addValue({
    name:'is_free',
    table: '@',
    tableField: 'gratuito',
    resultField: 'is_free_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'gratuito'
    }
}).addValue({
    name:'night_time',
    table: '@',
    tableField: 'noturno_curso_t',
    resultField: 'night_time_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'noturno_curso_t'
    }
}).addValue({
    name:'situation',
    table: '@',
    tableField: 'cod_situacao_curso',
    resultField: 'situacao_curso_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_situacao_curso'
    }
}).addValue({
    name:'finish',
    table: '@',
    tableField: 'concluinte',
    resultField: 'finish_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'concluinte'
    }
}).addValue({
    name:'year',
    table: '@',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'min_year',
    table: '@',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: '@',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: '@',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: '@',
        field: 'ano_censo'
    }
}).addValue({
    name: 'age_student_code',
    table: '@',
    tableField: 'idade_aluno_codigo',
    resultField: 'age_student_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'idade_aluno_codigo'
    }
}).addValue({
    name:'upper_turn',
    table: '@',
    tableField: 'cod_turno_aluno',
    resultField: 'upper_turn_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_turno_aluno'
    }
}).addValue({
    name:'ethnic_group_ies',
    table: '@',
    tableField: 'par_cod_cor_raca_aluno',
    resultField: 'ethnic_group_ies_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_cod_cor_raca_aluno'
    }
}).addValue({
    name:'student_deficiency',
    table: '@',
    tableField: 'par_aluno_deficiencia_transtorno_superdotacao',
    resultField: 'student_deficiency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_aluno_deficiencia_transtorno_superdotacao'
    }
}).addValue({
    name:'school_type',
    table: '@',
    tableField: 'par_tipo_escola_ensino_medio',
    resultField: 'school_type_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_tipo_escola_ensino_medio'
    }
}).addValue({
    name: 'gender_ies',
    table: '@',
    tableField: 'par_genero_aluno',
    resultField: 'gender_ies_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'par_genero_aluno'
    }
}).addValue({
    name: 'mesoregion',
    table: 'municipio',
    tableField: ['nome_mesorregiao', 'mesorregiao_id'],
    resultField: ['mesoregion_name', 'mesoregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'localoferta_cod_municipio',
        foreignTable: 'localoferta_ens_superior_matricula'
    }
}).addValue({
    name: 'microregion',
    table: 'municipio',
    tableField: ['nome_microrregiao', 'microrregiao_id'],
    resultField: ['microregion_name', 'microregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'localoferta_cod_municipio',
        foreignTable: 'localoferta_ens_superior_matricula'
    }
}).addValue({
    name: 'course',
    table: 'curso_ens_superior',
    tableField: 'nome_curso',
    resultField: 'course_name',
    where:{
        relation: '=',
        type: 'string',
        table:  'curso_ens_superior',
        field: 'nome_curso'
    },
    join:{
        primary: ['ano_censo', 'cod_curso'],
        foreign: ['ano_censo', 'cod_curso'],
        foreignTable: 'localoferta_ens_superior_matricula'
    }
});

universityEnrollmentApp.get('/', rqf.parse(), (req, res, next) => {
    if ("localoffer" in req.dims) {
        if ("university" in req.dims || "universityLocalOffer" in req.dims) {
            req.sql.from('localoferta_ens_superior_matricula')
                .field('curso_ens_superior.ano_censo', 'year')
                .field('COUNT(localoferta_ens_superior.cod_local_oferta)', 'total')
                .group('localoferta_ens_superior_matricula.ano_censo')
                .where('localoferta_ens_superior_matricula.cod_aluno_situacao = 2 OR localoferta_ens_superior_matricula.cod_aluno_situacao = 6 OR localoferta_ens_superior_matricula.matriculado = 1')
                .where('localoferta_ens_superior_matricula.cod_nivel_academico = 1')
                .order('localoferta_ens_superior_matricula.ano_censo')
                .order('localoferta_ens_superior.cod_local_oferta');
        } else {
            req.sql.from('localoferta_ens_superior_matricula')
                .field('localoferta_ens_superior_matricula.ano_censo', 'year')
                .field('COUNT(*)', 'total')
                .field('localoferta_ens_superior_matricula.cod_ies', 'university_id')
                .field('localoferta_ens_superior_matricula.nome_ies', 'university_name')
                .where('localoferta_ens_superior_matricula.cod_aluno_situacao = 2 OR localoferta_ens_superior_matricula.cod_aluno_situacao = 6 OR localoferta_ens_superior_matricula.matriculado = 1')
                .where('localoferta_ens_superior_matricula.cod_nivel_academico = 1')
                .group('localoferta_ens_superior_matricula.ano_censo')
                .group('localoferta_ens_superior_matricula.cod_ies')
                .group('localoferta_ens_superior_matricula.nome_ies')
                .order('localoferta_ens_superior_matricula.ano_censo')
                .order('localoferta_ens_superior_matricula.cod_local_oferta');
        }
    } else if (("state" in req.dims) || ("city" in req.dims) || ("region" in req.dims) || ("mesoregion" in req.dims) || ("microregion" in req.dims) ||
        ("state" in req.filter) || ("city" in req.filter) || ("region" in req.filter) || ("mesoregion" in req.filter) || ("microregion" in req.filter)) {
        req.sql.from('localoferta_ens_superior_matricula')
        .field('DISTINCT COUNT(*)', 'total')
        .field("'Brasil'", 'name')
        .field('localoferta_ens_superior_matricula.ano_censo', 'year')
        .where('localoferta_ens_superior_matricula.cod_aluno_situacao = 2 OR localoferta_ens_superior_matricula.cod_aluno_situacao = 6 OR localoferta_ens_superior_matricula.matriculado = 1')
        .where('localoferta_ens_superior_matricula.cod_nivel_academico = 1')
        .group('localoferta_ens_superior_matricula.ano_censo')
        .order('localoferta_ens_superior_matricula.ano_censo')
    } else if ("university" in req.dims || "universityLocalOffer" in req.dims) {
        req.sql.from('aluno_ens_superior')
        .field('COUNT(*)', 'total')
        .field("'Brasil'", 'name')
        .field('aluno_ens_superior.ano_censo', 'year')
        .group('aluno_ens_superior.cod_ies')
        .group('aluno_ens_superior.ano_censo')
        .where('aluno_ens_superior.cod_aluno_situacao = 2 OR aluno_ens_superior.cod_aluno_situacao = 6 OR aluno_ens_superior.matriculado = 1')
        .where('aluno_ens_superior.cod_nivel_academico = 1')
        .order('aluno_ens_superior.cod_ies')
        .order('aluno_ens_superior.ano_censo')
    } else {
        req.sql.from('localoferta_ens_superior_matricula')
        .field('COUNT(*)', 'total')
        .field("'Brasil'", 'name')
        .field('localoferta_ens_superior_matricula.ano_censo', 'year')
        .where('localoferta_ens_superior_matricula.cod_aluno_situacao = 2 OR localoferta_ens_superior_matricula.cod_aluno_situacao = 6 OR localoferta_ens_superior_matricula.matriculado = 1')
        .where('localoferta_ens_superior_matricula.cod_nivel_academico = 1')
        .group('localoferta_ens_superior_matricula.ano_censo')
        .order('localoferta_ens_superior_matricula.ano_censo')
    }
    next();
}, rqf.build(), query, (req, res, next) =>{ console.log(req.sql.toString()); next()}, id2str.transform(), addMissing(rqf), (req, res, next) => {
    if ('course' in req.dims){
        var total_course = req.result.reduce((total, cur) => {return total += cur.total}, 0)
        for (var course of req.result){
            course.percentage = Number((( course.total / total_course ) * 100).toFixed(2))
        }
    }
    next();
}, response('universityEnrollment'));

universityEnrollmentApp.get('/enter_situation', rqf.parse(), (req, res, next) => {
    req.sql.from('localoferta_ens_superior_matricula')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=2 AND localoferta_ens_superior_matricula.ingressante=1 THEN 1 ELSE 0 END)', 'cursando')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=6 AND localoferta_ens_superior_matricula.ingressante=1 THEN 1 ELSE 0 END)', 'concluinte')
    .field('SUM(CASE WHEN (localoferta_ens_superior_matricula.cod_aluno_situacao=4 OR localoferta_ens_superior_matricula.cod_aluno_situacao=5 OR localoferta_ens_superior_matricula.cod_aluno_situacao=7) AND localoferta_ens_superior_matricula.ingressante=1 THEN 1 ELSE 0 END)', 'evadido')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=3 AND localoferta_ens_superior_matricula.ingressante=1 THEN 1 ELSE 0 END)', 'trancado')
    .field('COUNT(*)', 'total')
    .field("'Brasil'", 'name')
    .field('localoferta_ens_superior_matricula.ano_censo', 'year')
    .where('localoferta_ens_superior_matricula.cod_nivel_academico=1')
    .where('localoferta_ens_superior_matricula.cod_grau_academico=2 OR localoferta_ens_superior_matricula.cod_grau_academico=4')
    .group('localoferta_ens_superior_matricula.ano_censo')
    .order('localoferta_ens_superior_matricula.ano_censo')
    next()
}, rqf.build(), query, (req, res, next) => {
    for (var res of req.result){
        res.cursando = Number(res.cursando);
        res.concluinte = Number(res.concluinte);
        res.evadido = Number(res.evadido);
        res.trancado = Number(res.trancado);
        res.total = res.cursando + res.concluinte + res.evadido + res.trancado
        res.taxa_evasao = Number( ((res.evadido/res.total) * 100).toFixed(2) )
    }
    next();
}, id2str.transform(), response('enterSituation'));

universityEnrollmentApp.get('/enrollment_situation', rqf.parse(), (req, res, next) => {
    req.sql.from('localoferta_ens_superior_matricula')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=2 THEN 1 ELSE 0 END)', 'cursando')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=6 THEN 1 ELSE 0 END)', 'concluinte')
    .field('SUM(CASE WHEN (localoferta_ens_superior_matricula.cod_aluno_situacao=4 OR localoferta_ens_superior_matricula.cod_aluno_situacao=5 OR localoferta_ens_superior_matricula.cod_aluno_situacao=7) THEN 1 ELSE 0 END)', 'evadido')
    .field('SUM(CASE WHEN localoferta_ens_superior_matricula.cod_aluno_situacao=3 THEN 1 ELSE 0 END)', 'trancado')
    .field('localoferta_ens_superior_matricula.ano_censo', 'year')
    .field("'Brasil'", 'name')
    .where('localoferta_ens_superior_matricula.cod_nivel_academico=1')
    .where('localoferta_ens_superior_matricula.cod_grau_academico=2 OR localoferta_ens_superior_matricula.cod_grau_academico=4')
    .group('localoferta_ens_superior_matricula.ano_censo')
    .order('localoferta_ens_superior_matricula.ano_censo')
    next()
}, rqf.build(),query, (req, res, next) => {
    for (var res of req.result){
        res.cursando = Number(res.cursando);
        res.concluinte = Number(res.concluinte);
        res.evadido = Number(res.evadido);
        res.trancado = Number(res.trancado);
        res.total = res.cursando + res.concluinte + res.evadido + res.trancado
    }
    
    next();
}, id2str.transform(), response('enrollmentSituation'));



module.exports = universityEnrollmentApp;
