/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');
const { result } = require('lodash');

const enrollmentApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

enrollmentApp.use(cache('15 day'));

let rqf = new ReqQueryFields();

// Complete range of the enrollments dataset.
// Returns a tuple of start and ending years of the complete enrollments dataset.
enrollmentApp.get('/year_range', (req, res, next) => {
    req.sql.from('matricula')
    .field('MIN(matricula.ano_censo)', 'start_year')
    .field('MAX(matricula.ano_censo)', 'end_year');
    next();
}, query, response('range'));

enrollmentApp.get('/years', (req, res, next) => {
    req.sql.from('matricula')
    .field('DISTINCT matricula.ano_censo', 'year');
    next();
}, query, response('years'));

enrollmentApp.get('/source', (req, res, next) => {
    req.sql.from('fonte')
    .field('fonte', 'source')
    .where('tabela = \'matricula\'');
    next();
}, query, response('source'));

enrollmentApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

enrollmentApp.get('/diff_location', (req, res, next) => {
    req.result = [
        {id: 0, name: "A escola não está em localidade diferenciada"},
        {id: 1, name: "Área de assentamento"},
        {id: 2, name: "Terra indígena"},
        {id: 3, name: "Terra remanescente de quilombos"},
    ];
    next();
}, response('diff_location'));

// Returns all school years available
enrollmentApp.get('/school_year', (req, res, next) => {
    req.result = [];
    for(let i = 11; i <= 71; ++i) {
        let obj = {
            id: i,
            name: id2str.schoolYear(i)
        };

        if(obj.name !== id2str.schoolYear(99)) {
            req.result.push(obj);
        }
    }
    req.result.push({
        id: 99,
        name: id2str.schoolYear(99)
    });
    next();
}, response('school_year'));

// Returns all school years available
enrollmentApp.get('/education_level', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 74; ++i) {
        let obj = {
            id: i,
            name: id2str.educationLevel(i)
        };

        if(obj.name !== id2str.educationLevel(99)) {
            req.result.push(obj);
        }
    }
    req.result.push({
        id: 99,
        name: id2str.educationLevel(99)
    });
    next();
}, response('education_level'));

// Returns all school years available
enrollmentApp.get('/education_level_mod', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 12; ++i) {
        if (i == 3 || i == 6)
            continue;
        
        req.result.push({
            id: i,
            name: id2str.educationLevelMod(i)
        });
    }
    req.result.push({
        id: 99,
        name: id2str.educationLevelMod(99)
    });
    next();
}, response('education_level_mod'));

enrollmentApp.get('/education_level_short', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não classificada'},
        {id: 1, name: 'Creche'},
        {id: 2, name: 'Pré-Escola'},
        {id: 3, name: 'Ensino Fundamental - anos iniciais'},
        {id: 4, name: 'Ensino Fundamental - anos finais'},
        {id: 5, name: 'Ensino Médio'},
        {id: 6, name: 'EJA'},
        {id: 7, name: 'EE exclusiva'}
    ];
    next();
}, response('education_level_short'));

// Returns all adm dependencies
enrollmentApp.get('/adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependency(i)
        });
    };
    next();
}, response('adm_dependency'));

enrollmentApp.get('/adm_dependency_detailed', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 8; ++i) {
        req.result.push({
            id: i,
            name: id2str.admDependencyPriv(i)
        });
    };
    next();
}, response('adm_dependency_detailed'));

// Return genders
enrollmentApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));

// Return ethnic group
enrollmentApp.get('/ethnic_group', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <=5; ++i) {
        req.result.push({
            id: i,
            name: id2str.ethnicGroup(i)
        });
    }
    next();
}, response('ethnic_group'));

enrollmentApp.get('/period', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.period(i)
        });
    }
    req.result.push({
        id: 99,
        name: id2str.period(99)
    });
    next();
}, response('period'));

// Returns integral-time avaible
enrollmentApp.get('/integral_time', (req, res, next) => {
    req.result = [];
    for(let i = 0; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.integralTime(i)
        });
    }
    next();
}, response('integral_time'));

enrollmentApp.get('/special_class', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não Declarado'},
        {id: 0, name: 'Não'},
        {id: 1, name: 'Sim'}
    ];
    next();
}, response('special_class'));

enrollmentApp.get('/pee', (req, res, next) => {
    req.result = [
        {id: true, name: id2str.booleanVariable(true)},
        {id: false, name: id2str.booleanVariable(false)}
    ];
    next();
}, response('pee'))

enrollmentApp.get('/pee_por_categoria', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 14; ++i) {
        req.result.push({
            id: i,
            name: id2str.peePorCategoria(i)
        });
    }
    next();
}, response('pee_por_categoria'));

enrollmentApp.get('/age_range_all', (req, res, next) => {
    req.result = [
        {id: 1, name: '0 a 3 anos'},
        {id: 2, name: '4 a 5 anos'},
        {id: 3, name: '6 a 10 anos'},
        {id: 4, name: '11 a 14 anos'},
        {id: 5, name: '15 a 17 anos'},
        {id: 6, name: '18 a 24 anos'},
        {id: 7, name: '25 a 29 anos'},
        {id: 8, name: '30 a 40 anos'},
        {id: 9, name: '41 a 50 anos'},
        {id: 10, name: '51 a 64 anos'},
        {id: 11, name: 'Mais que 64 anos'}
    ];
    next();
}, response('age_range_all'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'adm_dependency',
    table: 'matricula',
    tableField: 'dependencia_adm_id',
    resultField: 'adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_id'
    }
}).addValue({
    name: 'adm_dependency_detailed',
    table: 'matricula',
    tableField: 'dependencia_adm_priv',
    resultField: 'adm_dependency_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm_priv'
    }
}).addValue({
    name: 'school_year',
    table: 'matricula',
    tableField: 'serie_ano_id',
    resultField: 'school_year_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'serie_ano_id'
    }
}).addValue({
    name: 'education_level',
    table: 'matricula',
    tableField: 'etapa_ensino_id',
    resultField: 'education_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_ensino_id'
    }
}).addValue({
    name: 'education_level_mod',
    table: 'matricula',
    tableField: 'etapas_mod_ensino_segmento_id',
    resultField: 'education_level_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapas_mod_ensino_segmento_id'
    }
}).addValue({
    name: 'education_level_short',
    table: 'matricula',
    tableField: 'etapa_resumida',
    resultField: 'education_level_short_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'etapa_resumida'
    }
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'matricula'
    }
}).addValue({
    name: 'mesoregion',
    table: 'municipio',
    tableField: ['nome_mesorregiao', 'mesorregiao_id'],
    resultField: ['mesoregion_name', 'mesoregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'matricula'
    }
}).addValue({
    name: 'microregion',
    table: 'municipio',
    tableField: ['nome_microrregiao', 'microrregiao_id'],
    resultField: ['microregion_name', 'microregion_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'matricula'
    }
}).addValueToField({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'matricula'
    }
}, 'dims').addValueToField({
    name: 'state',
    table: 'estado',
    tableField: 'nome',
    resultField: 'state_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'matricula'
    }
}, 'filter').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: ['nome', 'id'],
    resultField: ['city_name', 'city_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'matricula'
    }
}, 'dims').addValueToField({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'matricula'
    }
}, 'filter').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: ['nome_escola', 'id'],
    resultField: ['school_name', 'school_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'matricula'
    }
}, 'dims').addValueToField({
    name: 'locale_id',
    table: 'matricula',
    tableField: 'localizacao_id',
    resultField: 'locale_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}, 'dims').addValueToField({
    name: 'school_id',
    table: 'escola',
    tableField: 'id',
    resultField: 'school_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'matricula'
    }
}, 'dims').addValueToField({
    name: 'school',
    table: 'escola',
    tableField: 'nome_escola',
    resultField: 'school_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: ['id', 'ano_censo'],
        foreign: ['escola_id', 'ano_censo'],
        foreignTable: 'matricula'
    }
}, 'filter').addValue({
    name: 'location',
    table: 'matricula',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'diff_location',
    table: 'matricula',
    tableField: 'localizacao_diferenciada_par',
    resultField: 'diff_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_diferenciada_par'
    }
}).addValue({
    name: 'rural_location',
    table: 'matricula',
    tableField: 'localidade_area_rural',
    resultField: 'rural_location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localidade_area_rural'
    }
}).addValue({
    name: 'min_year',
    table: 'matricula',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'matricula',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'gender',
    table: 'matricula',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'ethnic_group',
    table: 'matricula',
    tableField: 'cor_raca_id',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca_id'
    }
}).addValue({
    name: 'period',
    table: 'matricula',
    tableField: 'turma_turno_id',
    resultField: 'period_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'turma_turno_id'
    }
}).addValue({
  name:'integral_time',
  table: 'matricula',
  tableField: 'tempo_integral',
  resultField: 'integral_time_id',
  where: {
      relation: '=',
      type: 'integer',
      field: 'tempo_integral'
  }
}).addValue({
  name:'age_range_all',
  table: 'matricula',
  tableField: 'faixa_etaria_31_03',
  resultField: 'age_range_all_id',
  where: {
      relation: '=',
      type: 'integer',
      field: 'faixa_etaria_31_03'
  }
}).addValue({
  name:'special_class',
  table: 'matricula',
  tableField: 'exclusiva_especial',
  resultField: 'special_class_id',
  where: {
      relation: '=',
      type: 'boolean',
      field: 'exclusiva_especial'
  }
}).addValueToField({
    name: 'period_not',
    table: 'matricula',
    tableField: 'turma_turno_id',
    resultField: 'period_id',
    where: {
        relation: '<>',
        type: 'integer',
        field: 'turma_turno_id'
    }
}, 'filter')
.addValue({
    name: 'low_vision',
    table: 'matricula',
    tableField: 'baixa_visao',
    resultField: 'low_vision',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'baixa_visao'
    } 
}).addValue({
    name: 'blindness',
    table: 'matricula',
    tableField: 'cegueira',
    resultField: 'blindness',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'cegueira'
    }
}).addValue({
    name: 'deafness',
    table: 'matricula',
    tableField: 'surdez',
    resultField: 'deafness',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'surdez'
    }
}).addValue({
    name: 'hearing_deficiency',
    table: 'matricula',
    tableField: 'deficiencia_auditiva',
    resultField: 'hearing_deficiency',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'deficiencia_auditiva'
    }
}).addValue({
    name: 'deafblindness',
    table: 'matricula',
    tableField: 'surdo_cegueira',
    resultField: 'deafblindness',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'surdo_cegueira'
    }
}).addValue({
    name: 'physical_disability',
    table: 'matricula',
    tableField: 'deficiencia_fisica',
    resultField: 'physical_disability',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'deficiencia_fisica'
    }
}).addValue({
    name: 'intellectual_disability',
    table: 'matricula',
    tableField: 'deficiencia_intelectual',
    resultField: 'intellectual_disability',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'deficiencia_intelectual'
    }
}).addValue({
    name: 'multiple_disabilities',
    table: 'matricula',
    tableField: 'deficiencia_multiplas',
    resultField: 'multiple_disabilities',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'deficiencia_multiplas'
    }
}).addValue({
    name: 'autism',
    table: 'matricula',
    tableField: 'autismo',
    resultField: 'autism',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'autismo'
    }
}).addValue({
    name: 'autism_spectrum_disorder',
    table: 'matricula',
    tableField: 'transtorno_espectro_autista',
    resultField: 'autism_spectrum_disorder',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'transtorno_espectro_autista'
    }
}).addValue({
    name: 'asperger_syndrom',
    table: 'matricula',
    tableField: 'sindrome_asperger',
    resultField: 'asperger_syndrom',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'sindrome_asperger'
    }
}).addValue({
    name: 'rett_syndrom',
    table: 'matricula',
    tableField: 'sindrome_rett',
    resultField: 'rett_syndrom',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'sindrome_rett'
    }
}).addValue({
    name: 'childhood_desintegrative_disorder',
    table: 'matricula',
    tableField: 'transtorno_desintegrativo_da_infancia',
    resultField: 'childhood_desintegrative_disorder',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'transtorno_desintegrativo_da_infancia'
    }
}).addValue({
    name: 'supergifted',
    table: 'matricula',
    tableField: 'superdotado',
    resultField: 'supergifted',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'superdotado'
    }
}).addValue({
    name: 'pee',
    table: 'matricula',
    tableField: 'possui_necessidade_especial',
    resultField: 'pee_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'possui_necessidade_especial'
    }
}).addValue({
    name: 'pee_por_categoria',
    table: 'matricula',
    tableField: 'possui_necessidade_especial',
    resultField: 'pee_por_categoria',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'possui_necessidade_especial'
    }
});

enrollmentApp.get('/', rqf.parse(), (req, res, next) => {
    if('pee_por_categoria' in req.dims){
        delete req.dims.pee_por_categoria
        req.pee_por_categoria = true
        req.sql.field('SUM(CASE WHEN cegueira = true THEN 1 ELSE 0 END)', 'Cegueira')
        .field('SUM(CASE WHEN baixa_visao = true THEN 1 ELSE 0 END)', 'Baixa visão')
        .field('SUM(CASE WHEN surdez = true THEN 1 ELSE 0 END)', 'Surdez')
        .field('SUM(CASE WHEN deficiencia_auditiva = true THEN 1 ELSE 0 END)', 'Deficiência auditiva')
        .field('SUM(CASE WHEN surdo_cegueira = true THEN 1 ELSE 0 END)', 'Surdocegueira')
        .field('SUM(CASE WHEN deficiencia_fisica = true THEN 1 ELSE 0 END)', 'Deficiência física')
        .field('SUM(CASE WHEN deficiencia_intelectual = true THEN 1 ELSE 0 END)', 'Deficiência intelectual')
        .field('SUM(CASE WHEN deficiencia_multiplas = true THEN 1 ELSE 0 END)', 'Deficiências múltiplas')
        .field('SUM(CASE WHEN autismo = true THEN 1 ELSE 0 END)', 'Autismo')
        .field('SUM(CASE WHEN transtorno_espectro_autista = true THEN 1 ELSE 0 END)', 'Transtorno do Espectro Autista (TEA)')
        .field('SUM(CASE WHEN sindrome_asperger = true THEN 1 ELSE 0 END)', 'Síndrome de Asperger')
        .field('SUM(CASE WHEN sindrome_rett = true THEN 1 ELSE 0 END)', 'Síndrome de Rett')
        .field('SUM(CASE WHEN transtorno_desintegrativo_da_infancia = true THEN 1 ELSE 0 END)', 'Transtorno desintegrativo da infância')
        .field('SUM(CASE WHEN superdotado = true THEN 1 ELSE 0 END)', 'Altas habilidades / Superdotação')
        .field('matricula.ano_censo', 'year')
        .from('matricula')
        .group('matricula.ano_censo')
        .order('matricula.ano_censo')
        .where('((matricula.tipo<=3 OR matricula.tipo IS NULL) AND (matricula.tipo_atendimento_turma IS NULL OR matricula.tipo_atendimento_turma <= 2))');
    }
    else{
        req.sql.field('COUNT(*)', 'total')
        .field('matricula.ano_censo', 'year')
        .from('matricula')
        .group('matricula.ano_censo')
        .order('matricula.ano_censo')
        .where('((matricula.tipo<=3 OR matricula.tipo IS NULL) AND (matricula.tipo_atendimento_turma IS NULL OR matricula.tipo_atendimento_turma <= 2))');
    }
    next();
}, rqf.build(), query, id2str.transform(false), (req, res, next) => {
    if(req.pee_por_categoria === true){
        let result = req.result;
        let result_total = [];
        for (var j = 0;j < result.length;j++){
            let result_parcial = result[j];
            for (var i in result_parcial){
                if(i !== 'year'){
                    let obj = {};
                    obj.total = result_parcial[i];
                    i = i.replace(/"/g, '');
                    obj.pee_por_categoria_name = i;
                    obj.year = result_parcial.year;
                    result_total.push(obj);
                }
            }
        }
        req.result= result_total;
    }
    next();
}, response('enrollment'));

enrollmentApp.get('/diagnosis', rqf.parse(), (req, res, next) => {
    req.dims = {};
    req.dims.state = true;
    req.dims.city = true;
    req.dims.school_year = true;
    req.dims.location = true;
    req.dims.adm_dependency_detailed = true;

    req.sql.field('COUNT(*)', 'total')
    .field("'Brasil'", 'name')
    .field('matricula.ano_censo', 'year')
    .from('matricula')
    .group('matricula.ano_censo')
    .order('matricula.ano_censo')
    .where('matricula.tipo<=3');

    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    let enrollments = req.result;

    // Gera a relação etapa de ensino X ano escolar
    let educationSchoolYear = {};
    for(let i = 10; i < 80; ++i) {
        if(id2str.schoolYear(i) !== id2str.schoolYear(99)) {
            let educationLevelId = Math.floor(i/10);
            educationSchoolYear[i] = {
                id: educationLevelId,
                name: id2str.educationLevelShort(educationLevelId),
            };
        }
    }

    let result = [];
    let educationLevelSet = new Set();
    let schoolYearSet = new Set();
    let i = 0;
    while(i < enrollments.length) {
        let enrollment = enrollments[i];
        if(!educationSchoolYear[enrollment.school_year_id]) {
            ++i;
            continue;
        }
        let educationLevelHash = '' + enrollment.year + educationSchoolYear[enrollment.school_year_id].id + enrollment.city_id;
        let schoolYearHash = '' + enrollment.year + enrollment.school_year_id + enrollment.city_id;

        let currentEducation = null;
        // Busca ou cria a etapa de ensino adequada
        if(educationLevelSet.has(educationLevelHash)) {
            let j = 0;
            let edu = result[j];
            while(j < result.length && (edu.year != enrollment.year || edu.education_level_school_year_id != educationSchoolYear[enrollment.school_year_id].id)) {
                ++j;
                edu = result[j];
            }
            if(j >= result.length) --j;
            edu = result[j];

            currentEducation = edu;
        } else {
            educationLevelSet.add(educationLevelHash);
            let obj = {
                year: enrollment.year,
                name: enrollment.name,
                state_id: enrollment.state_id,
                state_name: enrollment.state_name,
                city_id: enrollment.city_id,
                city_name: enrollment.city_name,
                education_level_school_year_id: educationSchoolYear[enrollment.school_year_id].id,
                education_level_school_year_name: educationSchoolYear[enrollment.school_year_id].name,
                total: 0,
                adm_dependencies: [
                    {
                        adm_dependency_detailed_id: enrollment.adm_dependency_detailed_id,
                        adm_dependency_detailed_name: enrollment.adm_dependency_detailed_name,
                        total: 0
                    }
                ],
                locations: [
                    {
                        location_id: enrollment.location_id,
                        location_name: enrollment.location_name,
                        total: 0
                    }
                ]
            };

            result.push(obj);
            currentEducation = obj;
        }

        let currentSchoolYear = null;
        // Busca ou cria a série adequada
        if(schoolYearSet.has(schoolYearHash)) {
            let j = 0;
            let edu = result[j];
            while(j < result.length && (edu.year != enrollment.year || edu.education_level_school_year_id != enrollment.school_year_id)) {
                ++j;
                edu = result[j];
            }
            if(j >= result.length) --j;
            edu = result[j];

            currentSchoolYear = edu;
        } else {
            schoolYearSet.add(schoolYearHash);
            let obj = {
                year: enrollment.year,
                name: enrollment.name,
                state_id: enrollment.state_id,
                state_name: enrollment.state_name,
                city_id: enrollment.city_id,
                city_name: enrollment.city_name,
                education_level_school_year_id: enrollment.school_year_id,
                education_level_school_year_name: enrollment.school_year_name,
                total: 0,
                adm_dependencies: [
                    {
                        adm_dependency_detailed_id: enrollment.adm_dependency_detailed_id,
                        adm_dependency_detailed_name: enrollment.adm_dependency_detailed_name,
                        total: 0
                    }
                ],
                locations: [
                    {
                        location_id: enrollment.location_id,
                        location_name: enrollment.location_name,
                        total: 0
                    }
                ]
            };

            result.push(obj);
            currentSchoolYear = obj;
        }

        // Adiciona ao total
        currentEducation.total += enrollment.total;
        currentSchoolYear.total += enrollment.total;

        // Adiciona ao total da dependência administrativa
        let admDependencyIndex = 0;
        let admDependency = currentEducation.adm_dependencies[admDependencyIndex];
        while (admDependencyIndex < currentEducation.adm_dependencies.length && enrollment.adm_dependency_detailed_id > admDependency.adm_dependency_detailed_id) {
            ++admDependencyIndex;
            admDependency = currentEducation.adm_dependencies[admDependencyIndex];
        }
        if(admDependencyIndex >= currentEducation.adm_dependencies.length || admDependency.adm_dependency_detailed_id != enrollment.adm_dependency_detailed_id) { // não encontrou
            let obj = {
                adm_dependency_detailed_id: enrollment.adm_dependency_detailed_id,
                adm_dependency_detailed_name: enrollment.adm_dependency_detailed_name,
                total: 0
            }
            currentEducation.adm_dependencies.splice(admDependencyIndex, 0, obj);
            admDependency = obj;
        }
        admDependency.total += enrollment.total;

        admDependencyIndex = 0;
        admDependency = currentSchoolYear.adm_dependencies[admDependencyIndex];
        while (admDependencyIndex < currentSchoolYear.adm_dependencies.length && enrollment.adm_dependency_detailed_id > admDependency.adm_dependency_detailed_id) {
            ++admDependencyIndex;
            admDependency = currentSchoolYear.adm_dependencies[admDependencyIndex];
        }
        if(admDependencyIndex >= currentSchoolYear.adm_dependencies.length || admDependency.adm_dependency_detailed_id != enrollment.adm_dependency_detailed_id) { // não encontrou
            let obj = {
                adm_dependency_detailed_id: enrollment.adm_dependency_detailed_id,
                adm_dependency_detailed_name: enrollment.adm_dependency_detailed_name,
                total: 0
            }
            currentSchoolYear.adm_dependencies.splice(admDependencyIndex, 0, obj);
            admDependency = obj;
        }
        admDependency.total += enrollment.total;

        // Adiciona ao total da localidade
        let locationIndex = 0;
        let location = currentEducation.locations[locationIndex];
        while (locationIndex < currentEducation.locations.length && enrollment.location_id > location.location_id) {
            ++locationIndex;
            location = currentEducation.locations[locationIndex];
        }
        if(locationIndex >= currentEducation.locations.length || location.location_id != enrollment.location_id) {
            let obj = {
                location_id: enrollment.location_id,
                location_name: enrollment.location_name,
                total: 0
            }
            currentEducation.locations.splice(locationIndex, 0, obj);
            location = obj;
        }
        location.total += enrollment.total;

        locationIndex = 0;
        location = currentSchoolYear.locations[locationIndex];
        while (locationIndex < currentSchoolYear.locations.length && enrollment.location_id > location.location_id) {
            ++locationIndex;
            location = currentSchoolYear.locations[locationIndex];
        }
        if(locationIndex >= currentSchoolYear.locations.length || location.location_id != enrollment.location_id) {
            let obj = {
                location_id: enrollment.location_id,
                location_name: enrollment.location_name,
                total: 0
            }
            currentSchoolYear.locations.splice(locationIndex, 0, obj);
            location = obj;
        }
        location.total += enrollment.total;

        ++i;
    }

    req.result = result;

    next();
}, response('enrollment_diagnosis'));

enrollmentApp.get('/projection', rqf.parse(), (req, res, next) => {
    req.dims = {};
    req.dims.state = true;
    req.dims.city = true;
    req.dims.location = true;
    req.dims.school_year = true;
    req.dims.adm_dependency = true;
    req.dims.period = true;
    req.filter.adm_dependency = [1,2,3];

    req.sql.field('COUNT(*)', 'total')
    .field("'Brasil'", 'name')
    .field('matricula.ano_censo', 'year')
    .from('matricula')
    .group('matricula.ano_censo')
    .order('matricula.ano_censo')
    .where('matricula.tipo<=3');

    next();
}, rqf.build(), query, id2str.transform(), (req, res, next) => {
    let enrollments = req.result;

    // Gera a relação etapa de ensino X ano escolar
    let educationSchoolYear = {};
    for(let i = 10; i < 80; ++i) {
        if(id2str.schoolYear(i) !== id2str.schoolYear(99)) {
            let educationLevelId = Math.floor(i/10);
            educationSchoolYear[i] = {
                id: educationLevelId,
                name: id2str.educationLevelShort(educationLevelId),
            };
        }
    }

    let result = [];
    let educationLevelSet = new Set();
    let schoolYearSet = new Set();
    let i = 0;
    while(i < enrollments.length) {
        let enrollment = enrollments[i];
        if(!educationSchoolYear[enrollment.school_year_id]) {
            ++i;
            continue;
        }
        let educationLevelHash = '' + enrollment.year + educationSchoolYear[enrollment.school_year_id].id + enrollment.city_id;
        let schoolYearHash = '' + enrollment.year + enrollment.school_year_id + enrollment.city_id;

        let currentEducation = null;
        // Busca ou cria a etapa de ensino adequada
        if(educationLevelSet.has(educationLevelHash)) {
            let j = 0;
            let edu = result[j];
            while(j < result.length && (edu.year != enrollment.year || edu.education_level_school_year_id != educationSchoolYear[enrollment.school_year_id].id)) {
                ++j;
                edu = result[j];
            }
            if((j >= result.length)) --j;
            edu = result[j];

            currentEducation = edu;
        } else {
            educationLevelSet.add(educationLevelHash);
            let obj = {
                year: enrollment.year,
                name: enrollment.name,
                state_id: enrollment.state_id,
                state_name: enrollment.state_name,
                city_id: enrollment.city_id,
                city_name: enrollment.city_name,
                education_level_school_year_id: educationSchoolYear[enrollment.school_year_id].id,
                education_level_school_year_name: educationSchoolYear[enrollment.school_year_id].name,
                urban_day_total: 0,
                urban_night_total: 0,
                rural_day_total: 0,
                rural_night_total: 0
            };
            result.push(obj);
            currentEducation = obj;
        }

        let currentSchoolYear = null;
        // Busca ou cria a série adequada
        if(schoolYearSet.has(schoolYearHash)) {
            let j = 0;
            let edu = result[j];
            while(j < result.length && (edu.year != enrollment.year || edu.education_level_school_year_id != enrollment.school_year_id)){
                ++j;
                edu = result[j];
            }
            if(j >= result.length) --j;
            edu = result[j];

            currentSchoolYear = edu;
        } else {
            schoolYearSet.add(schoolYearHash);
            let obj = {
                year: enrollment.year,
                name: enrollment.name,
                state_id: enrollment.state_id,
                state_name: enrollment.state_name,
                city_id: enrollment.city_id,
                city_name: enrollment.city_name,
                education_level_school_year_id: enrollment.school_year_id,
                education_level_school_year_name: enrollment.school_year_name,
                urban_day_total: 0,
                urban_night_total: 0,
                rural_day_total: 0,
                rural_night_total: 0
            };

            result.push(obj);
            currentSchoolYear = obj;
        }

        if(enrollment.location_id == 1) {
            if(enrollment.period_id < 3) {
                currentEducation.urban_day_total += enrollment.total;
                currentSchoolYear.urban_day_total += enrollment.total;
            } else {
                currentEducation.urban_night_total += enrollment.total;
                currentSchoolYear.urban_night_total += enrollment.total;
            }
        } else {
            if(enrollment.period_id < 3) {
                currentEducation.rural_day_total += enrollment.total;
                currentSchoolYear.rural_day_total += enrollment.total;
            } else {
                currentEducation.rural_night_total += enrollment.total;
                currentSchoolYear.rural_night_total += enrollment.total;
            }
        }

        ++i;
    }

    req.result = result;

    next();
}, response('enrollment_projection'));

module.exports = enrollmentApp;
