/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const basicEducationConclusion = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

basicEducationConclusion.use(cache('15 day'));

basicEducationConclusion.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    .where('pnad_novo.ano_ref >= 2019')
    next();
}, query, response('years'));

basicEducationConclusion.get('/illiteracy', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 2; i++) {
        req.result.push({
            id: i, name: id2str.illiteracy(i)
        });

    }
    req.result.push({id: 9, name: id2str.illiteracy(9)});
    next();
}, response('illiteracy'));

basicEducationConclusion.get('/years_of_study', (req, res, next) => {
    req.result = []

    for (let i = 0; i < 17; i++) {
        req.result.push({
            id: i, name: id2str.yearsOfStudy(i)
        });
    }

    req.result.push({id: 99, name: id2str.yearsOfStudy(99)});
    next();
}, response('years_of_study'));

basicEducationConclusion.get('/instruction_level', (req, res, next) => {
    req.result = []

    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.instructionLevel(i)
        });
    }
    req.result.push({id: 99, name: id2str.instructionLevel(99)});
    next();
}, response('instruction_level'));

basicEducationConclusion.get('/new_pnad_adm_dependency', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.newPnadAdmDependency(i)
        });
    }
    req.result.push({id: 99, name: id2str.newPnadAdmDependency(99)});
    next();
}, response('new_pnad_adm_dependency'));

basicEducationConclusion.get('/region', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.regionCode(i)
        });
    }

    next();
}, response('region'));

basicEducationConclusion.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

basicEducationConclusion.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

basicEducationConclusion.get('/attended_modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 16; i++) {
        req.result.push({
            id: i, name: id2str.attendedModality(i)
        });
    }
    // Remove the option with id equals 10 => This option exists in the database, a better solution to this would be remove the option from the database
    req.result.splice(req.result.findIndex((item) => item.id === 10), 1);
    req.result.push({id: 99, name: id2str.attendedModality(99)});
    next();
}, response('attended_modality'));

basicEducationConclusion.get('/income_range', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.incomeRange(i)
        });
    }
    req.result.push({id: 10, name: id2str.incomeRange(10)});
    next();
}, response('income_range'));

basicEducationConclusion.get('/attends_school', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    next();
}, response('attends_school'));

basicEducationConclusion.get('/gender', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

basicEducationConclusion.get('/new_pnad_ethnic_group', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.ethnicGroupNewPnad(i)
        });
    }
    req.result.push({id: 9, name: id2str.ethnicGroupNewPnad(9)});
    next();
}, response('new_pnad_ethnic_group'));

basicEducationConclusion.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

basicEducationConclusion.get('/modality', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 4; i++) {
        req.result.push({
            id: i, name: id2str.modality(i)
        });
    }
    req.result.push({id: 99, name: id2str.modality(99)});
    next();
}, response('modality'));

basicEducationConclusion.get('/modality_shift', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.modalityShift(i)
        });
    }
    req.result.push({id: 9, name: id2str.modalityShift(9)});
    req.result.push({id: 99, name: id2str.modalityShift(99)});
    next();
}, response('modality_shift'));

basicEducationConclusion.get('/state', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.stateName(i) !== 'Não declarada') {
            req.result.push({
                id: i, name: id2str.stateName(i)
            });
        }
    }
    req.result.push({id: 99, name: id2str.stateName(99)});

    next();
}, response('state'));

basicEducationConclusion.get('/age_range_all', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 12; i++) {
        req.result.push({
            id: i, name: id2str.ageRangeAll(i)
        });
    }

    next();
}, response('age_range_all'));

basicEducationConclusion.get('/location', (req, res, next) => {
    req.result = []
    for (let i = 1; i <= 2; i++) {
        req.result.push({
            id: i, name: id2str.location(i)
        });
    }

    next();
}, response('location'));


rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'state_not',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '<>',
        type: 'integer',
        field: 'cod_uf',
        table: 'pnad_novo'
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'years_of_study',
    table: 'pnad_novo',
    tableField: 'anos_de_estudo',
    resultField: 'years_of_study_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'anos_de_estudo'
    }
}).addValue({
    name: 'instruction_level',
    table: 'pnad_novo',
    tableField: 'nivel_de_instrucao',
    resultField: 'instruction_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_de_instrucao'
    }
}).addValue({
    name: 'new_pnad_adm_dependency',
    table: 'pnad_novo',
    tableField: 'dependencia_adm',
    resultField: 'new_pnad_adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'dependencia_adm'
    }
}).addValue({
    name: 'attends_school',
    table: 'pnad_novo',
    tableField: 'frequenta_escola',
    resultField: 'attends_school_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'frequenta_escola'
    }
}).addValue({
    name: 'modality',
    table: 'pnad_novo',
    tableField: 'modalidade',
    resultField: 'modality_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'modalidade'
    }
}).addValue({
    name: 'attended_modality',
    table: 'pnad_novo',
    tableField: 'nivel_etapa_modalidade_freq',
    resultField: 'attended_modality_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'nivel_etapa_modalidade_freq'
    }
}).addValue({
    name: 'illiteracy',
    table: 'pnad_novo',
    tableField: 'analfabetismo',
    resultField: 'illiteracy_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'analfabetismo'
    }
}).addValue({
    name: 'modality_shift',
    table: 'pnad_novo',
    tableField: 'turno_nivel_etapa',
    resultField: 'modality_shift_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'turno_nivel_etapa'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'age_range_all',
    table: 'pnad_novo',
    tableField: 'faixa_etaria',
    resultField: 'age_range_all_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux_tx',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux_tx'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'location',
    table: 'pnad_novo',
    tableField: 'situacao_domicilio',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'situacao_domicilio'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addField({
    name: 'search',
    field: false,
    where: true
}).addValueToField({
    name: 'name',
    table: 'pnad_novo',
    tableField: 'nome',
    where: {
        relation: 'LIKE',
        type: 'string',
        field: 'nome'
    }
}, 'search').addValue({
    name: 'mesoregion',
    table: 'pnad_novo',
    tableField: 'mesorregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id'
    }
}).addValue({
    name: 'microregion',
    table: 'pnad_novo',
    tableField: 'microrregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id'
    }
});

function matchQueries(queryPartial, queryTotal) {
    let match = [];
    queryTotal.forEach((result) => {
        let newObj = {};
        let keys = Object.keys(result);
        keys.forEach((key) => {
            newObj[key] = result[key];
        });
        let index = keys.indexOf('total');
        if(index > -1) keys.splice(index, 1);
        let objMatch = null;

        for(let i = 0; i < queryPartial.length; ++i) {
            let partial = queryPartial[i];
            let foundMatch = true;
            for(let j = 0; j < keys.length; ++j) {
                let key = keys[j];
                if(partial[key] !== result[key]) {
                    foundMatch = false;
                    break;
                }
            }
            if(foundMatch) {
                objMatch = partial;
                break;
            }
        }

        if(objMatch) {
            newObj.denominator = result.total;
            newObj.partial = objMatch.total;
            newObj.total = (objMatch.total / result.total) * 100;
            match.push(newObj);
        }
    });

    return match;
}

basicEducationConclusion.get('/', rqf.parse(), rqf.build(), (req, res, next) => {
    req.querySet = [];
    
    // Subquery para total_pop_maior_19 com filtros dinâmicos
    let totalPopMaior19 = req.sql.clone();
        totalPopMaior19.from("pnad_novo")
        .field("ano_ref", "year")
        .field("SUM(peso_domicilio_pessoas_com_cal)", "total")
        .where("idade_morador_31_03 >= 19")
        .where("ano_ref >= 2019")
        .group("ano_ref")
        .order("ano_ref");

    // Subquery para total_pop_ed_bas_maior_19 com filtros dinâmicos
    let totalPopEdBasMaior19 = req.sql.clone();
        totalPopEdBasMaior19.from("pnad_novo")
        .field("ano_ref", "year")
        .field("SUM(peso_domicilio_pessoas_com_cal)", "total")
        .where("idade_morador_31_03 >= 19")
        .where("nivel_de_instruc_mais_elevad_para_o_fundam_com_duracao_9_anos IN (5, 6, 7)")
        .where("ano_ref >= 2019")
        .group("ano_ref")
        .order("ano_ref");

    if (req.query.dims && req.query.dims.includes("income_range")) {
        totalPopMaior19.where("faixa_rendimento_aux_tx is not null");
        totalPopEdBasMaior19.where("faixa_rendimento_aux_tx is not null");
    }

    req.querySet.push(totalPopEdBasMaior19);
    req.querySet.push(totalPopMaior19);

    next();
}, multiQuery, (req, res, next) => {
        // The multiple requests are made. Then we need to calculate the percetange. So the function
        // below is used
        let newObj = matchQueries(req.result[0], req.result[1]);
        req.result = newObj;
    next();
}, id2str.transform(false), response('basic_education_conclusion'));


module.exports = basicEducationConclusion;
