/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const CourseAggregateApp = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

CourseAggregateApp.use(cache('15 day'));

CourseAggregateApp.get('/years', (req, res, next) => {
    req.sql.from('curso_superior_agregado')
    .field('DISTINCT curso_superior_agregado.ano_censo', 'year')
    next();
}, query, response('years'));

CourseAggregateApp.get('/upper_adm_dependency', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 7; ++i) {
        req.result.push({
            id: i,
            name: id2str.upperAdmDependency(i)
        });
    };
    next();
}, response('upper_adm_dependency'));

CourseAggregateApp.get('/academic_organization', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 5; ++i) {
        req.result.push({
            id: i,
            name: id2str.academicOrganization(i)
        });
    };
    next();
}, response('academic_organization'));

CourseAggregateApp.get('/academic_level', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 4; ++i) {
        req.result.push({
            id: i,
            name: id2str.academicLevel(i)
        });
    };
    next();
}, response('academic_level'));

CourseAggregateApp.get('/upper_education_mod', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 2; ++i) {
        req.result.push({
            id: i,
            name: id2str.upperEducationMod(i)
        });
    };
    next();
}, response('upper_education_mod'));

CourseAggregateApp.get('/region', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.regionCode(i)
        });
    }

    next();
}, response('region'));

CourseAggregateApp.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

CourseAggregateApp.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

CourseAggregateApp.get('/is_free', (req, res, next) => {
    req.result = [
        {id: null, name: 'Não Classificado'},
        {id: 0, name: 'Não'},
        {id: 1, name: 'Sim'}
    ];
    next();
}, response('is_free'));


CourseAggregateApp.get('/cine_geral', (req, res, next) => {
    req.result = [];
    for(let i = 1; i <= 10; ++i) {
        req.result.push({
            id: i,
            name: id2str.cineGeral(i)
        });
    };
    next();
}, response('cine_geral'));

CourseAggregateApp.get('/cine_specific', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 1; i <= 104; ++i) {
        let obj = {
            id: i,
            name: id2str.cineSpecific(i)
        };
        if (obj.name !== id2str.cineSpecific(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.cineSpecific(defaultCase)
    });
    next();
}, response('cine_specific'));

CourseAggregateApp.get('/cine_detailed', (req, res, next) => {
    req.result = [];
    const defaultCase = null;
    for(let i = 11; i <= 1041; ++i) {
        let obj = {
            id: i,
            name: id2str.cineDetailed(i)
        };
        if (obj.name !== id2str.cineDetailed(defaultCase)){
            req.result.push(obj);
        }
    };
    req.result.push({
        id: defaultCase,
        name: id2str.cineDetailed(defaultCase)
    });
    next();
}, response('cine_detailed'));

CourseAggregateApp.get('/university', (req, res, next) => {
    req.sql.from('curso_superior_agregado')
    .field('DISTINCT curso_superior_agregado.cod_ies', 'cod')
    .field('ies_ens_superior.nome_ies', 'nome')
    .join('ies_ens_superior', null, 'curso_superior_agregado.cod_ies = ies_ens_superior.cod_ies and curso_superior_agregado.ano_censo = ies_ens_superior.ano_censo')
    next();
}, query, response('university'));

CourseAggregateApp.get('/state', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.stateName(i) !== 'Não declarada') {
            req.result.push({
                id: i, name: id2str.stateName(i)
            });
        }
    }
    req.result.push({id: 99, name: id2str.stateName(99)});

    next();
}, response('state'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'upper_adm_dependency',
    table: 'curso_superior_agregado',
    tableField: 'tp_categ_adm',
    resultField: 'upper_adm_dependency_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tp_categ_adm'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'curso_superior_agregado'
    }
}).addValue({
    name: 'academic_organization',
    table: 'curso_superior_agregado',
    tableField: 'tp_org_acad',
    resultField: 'academic_organization_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tp_org_acad'
    }
}).addValue({
    name: 'academic_level',
    table: 'curso_superior_agregado',
    tableField: 'tp_grau_acad',
    resultField: 'academic_level_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tp_grau_acad'
    }
}).addValue({
    name: 'upper_education_mod',
    table: 'curso_superior_agregado',
    tableField: 'tp_modal_ens',
    resultField: 'upper_education_mod_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'tp_modal_ens'
    }
}).addValue({
    name: 'is_free',
    table: 'curso_superior_agregado',
    tableField: 'in_gratuito',
    resultField: 'is_free_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'in_gratuito'
    }
}).addValue({
    name: 'cine_geral',
    table: 'curso_superior_agregado',
    tableField: 'cod_cine_area_geral',
    resultField: 'cine_geral_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_geral'
    }
}).addValue({
    name: 'cine_specific',
    table: 'curso_superior_agregado',
    tableField: 'cod_cine_area_esp',
    resultField: 'cine_specific_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_esp'
    }
}).addValue({
    name: 'cine_detailed',
    table: 'curso_superior_agregado',
    tableField: 'cod_cine_area_detalhada',
    resultField: 'cine_detailed_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cine_area_detalhada'
    }
}).addValue({
    name: 'uni_offer_aggregate',
    table: 'ies_ens_superior',
    tableField: ['cod_ies', 'nome_ies'],
    resultField: ['university_id', 'university_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_ies'
    },
    join: {
        primary: ['cod_ies', 'ano_censo'],
        foreign: ['cod_ies', 'ano_censo'],
        foreignTable: 'curso_superior_agregado'
    }
}).addValue({
    name: 'cap_code',
    table: 'curso_superior_agregado',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'region',
    table: 'curso_superior_agregado',
    tableField: 'cod_reg',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_reg'
    }
}).addValue({
    name: 'metro_code',
    table: 'curso_superior_agregado',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'min_year',
    table: 'curso_superior_agregado',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'curso_superior_agregado',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_censo'
    }
}).addValue({
    name: 'city',
    table: 'municipio',
    tableField: ['id', 'nome'],
    resultField: ['city_id', 'city_name'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
        table: 'municipio'
    },
    join: {
        primary: 'id',
        foreign: 'cod_mun',
        foreignTable: 'curso_superior_agregado'
    }
});

CourseAggregateApp.get('/', rqf.parse(), rqf.build(),  (req, res, next) => {
    req.sql.from('curso_superior_agregado')
    .field('count(distinct(cod_curso))', 'total')
    .field('curso_superior_agregado.ano_censo', 'year')
    .where('curso_superior_agregado.tp_nivel_acad = 1 AND qtd_cursos = 1')
    .group('curso_superior_agregado.ano_censo')
    .order('curso_superior_agregado.ano_censo')
    console.log(req.sql.toString())
    next();
}, query, id2str.transform(false), response('course_aggregate'));

module.exports = CourseAggregateApp;
