/*
Copyright (C) 2016 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const outOfSchoolApp = express.Router();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const addMissing = require(`${libs}/middlewares/addMissing`);

const config = require(`${libs}/config`);

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

outOfSchoolApp.use(cache('15 day'));

outOfSchoolApp.get('/year_range', (req, res, next) => {
    req.sql.from('pnad')
    .field('MIN(pnad.ano_censo)', 'start_year')
    .field('MAX(pnad.ano_censo)', 'end_year')
    .where('pnad.ano_censo >= 2007 AND pnad.ano_censo <= 2015');
    next();
}, query, response('range'));

outOfSchoolApp.get('/years', (req, res, next) => {
    req.sql.from('pnad').
    field('DISTINCT pnad.ano_censo', 'year')
    .where('pnad.ano_censo >= 2007 AND pnad.ano_censo <= 2015');
    next();
}, query, response('years'));

outOfSchoolApp.get('/full_age_range', (req, res, next) => {
    req.result = [
        {id: 1, name: '0 a 3 anos'},
        {id: 2, name: '4 a 5 anos'},
        {id: 3, name: '6 a 10 anos'},
        {id: 4, name: '11 a 14 anos'},
        {id: 5, name: '15 a 17 anos'},
        {id: 6, name: '18 a 24 anos'},
        {id: 7, name: '25 a 29 anos'},
        {id: 8, name: '30 a 40 anos'},
        {id: 9, name: '41 a 50 anos'},
        {id: 10, name: '51 a 64 anos'},
        {id: 11, name: 'Mais de 64 anos'}
    ];
    next();
}, response('full_age_range'));

outOfSchoolApp.get('/ethnic_group', (req, res, next) => {
    req.result = [
        {id: 0, name: 'Sem declaração'},
        {id: 1, name: 'Branca'},
        {id: 2, name: 'Preta'},
        {id: 3, name: 'Parda'},
        {id: 4, name: 'Amarela'},
        {id: 5, name: 'Indígena'}
    ];
    next();
}, response('ethnic_group'));

outOfSchoolApp.get('/location', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Urbana'},
        {id: 2, name: 'Rural'}
    ];
    next();
}, response('location'));

outOfSchoolApp.get('/gender', (req, res, next) => {
    req.result = [
        {id: 1, name: 'Masculino'},
        {id: 2, name: 'Feminino'}
    ];
    next();
}, response('gender'));

outOfSchoolApp.get('/fifth_household_income', (req, res, next) => {
    req.result = [
		{id: 1, name: '20% menores'},
        {id: 2, name: '2o quinto'},
        {id: 3, name: '3o quinto'},
        {id: 4, name: '4o quinto'},
		{id: 5, name: '20% maiores'},
        {id: -1, name: 'Sem declaração'}
	];
    next();
},response('fifth_household_income'));

outOfSchoolApp.get('/extremes_household_income', (req, res, next) => {
    req.result = [
		{id: 1, name: '10% menores'},
        {id: 2, name: '10% maiores'},
        {id: -1, name: 'Sem declaração'}
	];
    next();
}, response('extremes_household_income'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'region',
    table: 'regiao',
    tableField: ['nome', 'id'],
    resultField: ['region_name', 'region_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'regiao_id',
        foreignTable: 'pnad'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'pnad'
    }
}).addValue({
    name: 'ethnic_group',
    table: 'pnad',
    tableField: 'cor_raca_id',
    resultField: 'ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca_id'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: 'pnad',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: 'pnad',
        field: 'ano_censo'
    }
}).addValue({
    name: 'full_age_range',
    table: 'pnad',
    tableField: 'faixa_etaria_31_03',
    resultField: 'full_age_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria_31_03'
    }
}).addValue({
    name: 'gender',
    table: 'pnad',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'location',
    table: 'pnad',
    tableField: 'localizacao_id',
    resultField: 'location_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'localizacao_id'
    }
}).addValue({
    name: 'extremes_household_income',
    table: 'pnad',
    tableField: 'extremos_nivel_rendimento',
    resultField: 'extremes_household_income_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'extremos_nivel_rendimento'
    }
}).addValue({
    name: 'fifth_household_income',
    table: 'pnad',
    tableField: 'quintil_nivel_rendimento',
    resultField: 'fifth_household_income_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'quintil_nivel_rendimento'
    }
});

outOfSchoolApp.get('/', rqf.parse(), rqf.build(), (req, res, next) => {
    req.sql.from('pnad')
    .field('SUM(pnad.peso)', 'total')
    .field('pnad.ano_censo', 'year')
    .where('pnad.escolaridade_familiar >= 1 AND pnad.escolaridade_familiar <= 4 AND pnad.frequenta_escola_creche = 4')
    .group('pnad.ano_censo')
    .order('pnad.ano_censo');

    next();
}, query, addMissing(rqf), id2str.transform(), response('out_of_school'));

// Versão para o SimCAQ
let simcaqRqf = new ReqQueryFields();

simcaqRqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'brazil_total',
    table: 'populacao_fora_da_escola',
    tableField: 'brasil',
    resultField: 'brazil_total_id',
    where: {
        relation: '=',
        type: 'boolean',
        field: 'brasil'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: 'nome',
    resultField: 'state_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'estado_id',
        foreignTable: 'populacao_fora_da_escola'
    }
}).addValue({
    name: 'city',
    table: 'municipio',
    tableField: 'nome',
    resultField: 'city_name',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    },
    join: {
        primary: 'id',
        foreign: 'municipio_id',
        foreignTable: 'populacao_fora_da_escola'
    }
}).addValue({
    name: 'pfe',
    table: 'populacao_fora_da_escola',
    tableField: 'codigo_pfe',
    resultField: 'pfe_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'codigo_pfe'
    }
}).addValue({
    name: 'min_year',
    table: 'populacao_fora_da_escola',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        table: 'populacao_fora_da_escola',
        field: 'ano_censo'
    }
}).addValue({
    name: 'max_year',
    table: 'populacao_fora_da_escola',
    tableField: 'ano_censo',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        table: 'populacao_fora_da_escola',
        field: 'ano_censo'
    }
});

outOfSchoolApp.get('/simcaq', simcaqRqf.parse(), (req, res, next) => {
    if ('state' in req.filter || 'city' in req.filter || 'state' in req.dims || 'city' in req.dims) { // Query in state/city level

        if ('city' in req.filter && 'state' in req.filter) delete req.filter.state   // use only the city filter because of the table particularities

        req.sql.from('populacao_fora_da_escola')
        .field('SUM(populacao_fora_da_escola.pop_fora_escola)', 'total')
        .field("'Brasil'", 'name')
        .field('populacao_fora_da_escola.ano_censo')
        .group('populacao_fora_da_escola.ano_censo')
        .order('populacao_fora_da_escola.ano_censo');
    } else { // Query in 'Brasil' level
        req.sql.from('populacao_fora_da_escola')
        .field('SUM(populacao_fora_da_escola.pop_fora_escola)', 'total')
        .field("'Brasil'", 'name')
        .field('populacao_fora_da_escola.ano_censo')
        .where('populacao_fora_da_escola.brasil = 1')
        .group('populacao_fora_da_escola.ano_censo')
        .order('populacao_fora_da_escola.ano_censo'); 
    }

    next();
}, simcaqRqf.build(), query, (req, res, next) => {
	req.result.forEach((i) => {
		i.total = parseInt(i.total);
	});
	next();
}, addMissing(simcaqRqf), id2str.transform(), response('out_of_school'));

module.exports = outOfSchoolApp;
