const express = require('express');

const simulationApp = express();

const libs = `${process.cwd()}/libs`;

const log = require(`${libs}/log`)(module);

const Simulation = require(`${libs}/models/simulation`);

const PQR = require(`${libs}/models/pqr`);

const passport = require('passport');

simulationApp.get('/time', (req, res, next) => {
    const maxTime = parseInt(req.query.max_time, 10);
    if (isNaN(maxTime)) {
        res.status(400);
        next({
            status: 400,
            message: 'Invalid value for mandatory parameter max_time'
        });
    }
    res.json({
        result: Array.apply(null, { length: maxTime }).map(Number.call, Number).map((i) => i + 1)
    });
});

simulationApp.get('/pqr', async (req, res) => {
    let pqr = await PQR.findOne({ attributes: ['content'] }).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });
    if (pqr)
        res.json(pqr);
});

simulationApp.put('/pqr', passport.authenticate('bearer', { session: false }), async (req, res, next) => {
    let user = req.user;
    let pqr = await PQR.findOne().catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });
    if (!user.admin) {
        log.info(`Usuário ${user.email} tentou alterar o PQR, mas não tem privilégio`);
        res.statusCode = 401;
        return next({ err: { msg: 'Unauthorized' } });
    }

    if(pqr){
        let _content = req.body.content || pqr.content;
        pqr.content = _content;
        await pqr.save({fields:['content']}).catch(function (err) {
            if (err) {
                log.error(err);
                return next({ err });
            }
        });}
    
    res.json({ msg: 'PQR updated' })

});

simulationApp.get('/', passport.authenticate('bearer', { session: false }), async (req, res) => {
    let user = req.user;
    let simulations = await Simulation.findAll({ where: { user_id: user.id } }, { attributes: ['user_id', 'name', 'created_at', 'updated_at'] }).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    })
    if (!simulations){
        res.json("Simulations not found");
    }
    res.json(simulations);

    // Simulation.find({userId: user._id}, (err, simulations) => {
    //     if(err) {
    //         log.error(err);
    //         return next({err});
    //     }

    //     res.json(simulations);
    // });
});

simulationApp.post('/', passport.authenticate('bearer', { session: false }), async (req, res, next) => {
    let user = req.user;

    let simulation = await Simulation.create({
        user_id: user.id,
        content: req.body.content,
        name: req.body.name
    });

    await simulation.save().catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });

    res.json({ msg: 'Simulation created', simulation });
})

simulationApp.get('/:id', passport.authenticate('bearer', { session: false }), async (req, res) => {
    let user = req.user;

    let simulation = await Simulation.findOne({where:{id: req.params.id, user_id: user.id }}).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });
    res.json(simulation);

});

simulationApp.put('/:id', passport.authenticate('bearer', { session: false }), async (req, res, next) => {
    let user = req.user;

    let simulation = await Simulation.findOne({ where: { id: req.params.id, user_id: user.id } }).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });

    if (!simulation) {
        res.statusCode = 404;
        return next({ err: { msg: 'Simulation not found' } });
    }

    simulation.content = req.body.content || simulation.content;
    simulation.name = req.body.name || simulation.name;
    simulation.updated_at = new Date();
    await simulation.save({fields:['content','name','updated_at']}).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });
    res.json({ msg: 'Simulation created', simulation });
});

simulationApp.delete('/:id', passport.authenticate('bearer', { session: false }), async (req, res, next) => {
    let user = req.user;
    await Simulation.destroy({where: { "id": req.params.id, "user_id": user.id } }).catch(function (err) {
        if (err) {
            log.error(err);
            return next({ err });
        }
    });
    res.json({ msg: 'Simulation removed' });
});

module.exports = simulationApp;
