/*
Copyright (C) 2024 Centro de Computacao Cientifica e Software Livre
Departamento de Informatica - Universidade Federal do Parana - C3SL/UFPR

This file is part of simcaq-node.

simcaq-node is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

simcaq-node is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with simcaq-node.  If not, see <https://www.gnu.org/licenses/>.
*/

const express = require('express');

const adjustedLiquidFrequency = express.Router();

const libs = `${process.cwd()}/libs`;

const squel = require('squel');

const query = require(`${libs}/middlewares/query`).query;

const multiQuery = require(`${libs}/middlewares/multiQuery`);

const response = require(`${libs}/middlewares/response`);

const ReqQueryFields = require(`${libs}/middlewares/reqQueryFields`);

const id2str = require(`${libs}/middlewares/id2str`);

const config = require(`${libs}/config`); 

const cache = require('apicache').options({ debug: config.debug, statusCodes: {include: [200]} }).middleware;

let rqf = new ReqQueryFields();

adjustedLiquidFrequency.use(cache('15 day'));

adjustedLiquidFrequency.get('/years', (req, res, next) => {
    req.sql.from('pnad_novo')
    .field('DISTINCT pnad_novo.ano_ref', 'year')
    .where('pnad_novo.ano_ref >= 2019')
    next();
}, query, response('years'));

adjustedLiquidFrequency.get('/income_range', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 8; i++) {
        req.result.push({
            id: i, name: id2str.incomeRange(i)
        });
    }
    req.result.push({id: 10, name: id2str.incomeRange(10)});
    next();
}, response('income_range'));

adjustedLiquidFrequency.get('/cap_code', (req, res, next) => {
    req.result = []
    for (let i = 11; i < 54; i++) {
        if (id2str.capitalCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.capitalCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.capitalCode(99)});

    next();
}, response('cap_code'));

adjustedLiquidFrequency.get('/metro_code', (req, res, next) => {
    req.result = []
    for (let i = 13; i < 53; i++) {
        if (id2str.metroCode(i) !== 'Não informado') {
        req.result.push({
            id: i, name: id2str.metroCode(i)
        });
        }
    }
    req.result.push({id: 99, name: id2str.metroCode(99)});

    next();
}, response('metro_code'));

adjustedLiquidFrequency.get('/gender', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.gender(i)
        });
    }
    next();
}, response('gender'));

adjustedLiquidFrequency.get('/new_pnad_ethnic_group', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 6; i++) {
        req.result.push({
            id: i, name: id2str.ethnicGroupNewPnad(i)
        });
    }
    req.result.push({id: 9, name: id2str.ethnicGroupNewPnad(9)});
    next();
}, response('new_pnad_ethnic_group'));

adjustedLiquidFrequency.get('/age_range_all', (req, res, next) => {
    req.result = []
    for (let i = 1; i <= 6; i++) {
        req.result.push({
            id: i, name: id2str.ageRangeAll(i)
        });
    }
    next()
}, response('age_range_all'));

adjustedLiquidFrequency.get('/bolsa_familia', (req, res, next) => {
    req.result = []
    for (let i = 1; i < 3; i++) {
        req.result.push({
            id: i, name: id2str.attendsSchool(i)
        });
    }
    req.result.push({id: 9, name: id2str.attendsSchool(9)});
    next();
}, response('bolsa_familia'));

rqf.addField({
    name: 'filter',
    field: false,
    where: true
}).addField({
    name: 'dims',
    field: true,
    where: false
}).addValue({
    name: 'id',
    table: 'pnad_novo',
    tableField: 'id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'id'
    }
}).addValue({
    name: 'state',
    table: 'estado',
    tableField: ['id', 'nome'],
    resultField: ['state_id', 'state_nome'],
    where: {
        relation: '=',
        type: 'integer',
        field: 'id',
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'state_not',
    table: 'estado',
    tableField: ['nome', 'id'],
    resultField: ['state_name', 'state_id'],
    where: {
        relation: '<>',
        type: 'integer',
        field: 'cod_uf',
        table: 'pnad_novo'
    },
    join: {
        primary: 'id',
        foreign: 'cod_uf',
        foreignTable: 'pnad_novo'
    }
}).addValue({
    name: 'cap_code',
    table: 'pnad_novo',
    tableField: 'cod_cap',
    resultField: 'cap_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_cap'
    }
}).addValue({
    name: 'bolsa_familia',
    table: 'pnad_novo',
    tableField: 'recebeu_rendimentos_de_programa_bolsa_familia',
    resultField: 'bolsa_familia_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'recebeu_rendimentos_de_programa_bolsa_familia'
    }
}).addValue({
    name: 'new_pnad_ethnic_group',
    table: 'pnad_novo',
    tableField: 'cor_raca',
    resultField: 'new_pnad_ethnic_group_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cor_raca'
    }
}).addValue({
    name: 'income_range',
    table: 'pnad_novo',
    tableField: 'faixa_rendimento_aux',
    resultField: 'income_range_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_rendimento_aux'
    }
}).addValue({
    name: 'age_range_all',
    table: 'pnad_novo',
    tableField: 'faixa_etaria',
    resultField: 'age_range_all_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'faixa_etaria'
    }
}).addValue({
    name: 'gender',
    table: 'pnad_novo',
    tableField: 'sexo',
    resultField: 'gender_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'sexo'
    }
}).addValue({
    name: 'metro_code',
    table: 'pnad_novo',
    tableField: 'cod_rm_ride',
    resultField: 'metro_code_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_rm_ride'
    }
}).addValue({
    name: 'region',
    table: 'pnad_novo',
    tableField: 'cod_regiao',
    resultField: 'region_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'cod_regiao'
    }
}).addValue({
    name: 'min_year',
    table: 'pnad_novo',
    tableField: 'ano_ref',
    resultField: 'year',
    where: {
        relation: '>=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addValue({
    name: 'max_year',
    table: 'pnad_novo',
    tableField: '',
    resultField: 'year',
    where: {
        relation: '<=',
        type: 'integer',
        field: 'ano_ref'
    }
}).addField({
    name: 'search',
    field: false,
    where: true
}).addValueToField({
    name: 'name',
    table: 'pnad_novo',
    tableField: 'nome',
    where: {
        relation: 'LIKE',
        type: 'string',
        field: 'nome'
    }
}, 'search').addValue({
    name: 'mesoregion',
    table: 'pnad_novo',
    tableField: 'mesorregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'mesorregiao_id'
    }
}).addValue({
    name: 'microregion',
    table: 'pnad_novo',
    tableField: 'microrregiao_id',
    where: {
        relation: '=',
        type: 'integer',
        field: 'microrregiao_id'
    }
});

adjustedLiquidFrequency.get('/', rqf.parse(), (req, res, next) => {    
    const dimensions = req.dims || {}; // Obter dimensões solicitadas pela requisição

    if (!dimensions.age_range_all) {
        req.result = { error: "age_range_all should be selected" };
        next();
    }
    
    // Subquery: total_pop
    let totalPop = squel.select()
    .field("ano_ref")
    .field("faixa_etaria")
    .field("SUM(peso_domicilio_pessoas_com_cal)", "total")
    .from("pnad_novo")
    .where("faixa_etaria <= 6")
    .group("ano_ref")
    .group("faixa_etaria");
    
    
    // Aplicar filtros dinâmicos à subquery total_pop
    totalPop = rqf.buildQuery(req, totalPop);
    
    const joinQuery = `
        total_pop.ano_ref = pnad_novo.ano_ref AND
        total_pop.faixa_etaria = pnad_novo.apoio_frequencia_ajustada
        ${dimensions.gender ? " AND total_pop.gender_id = pnad_novo.sexo" : ""}
        ${dimensions.bolsa_familia ? "AND total_pop.bolsa_familia_id = pnad_novo.recebeu_rendimentos_de_programa_bolsa_familia" : ""}
        ${dimensions.new_pnad_ethnic_group ? "AND total_pop.new_pnad_ethnic_group_id = pnad_novo.cor_raca" : ""}
        ${dimensions.income_range ? "AND total_pop.income_range_id = pnad_novo.faixa_rendimento_aux" : ""}
        ${dimensions.cap_code ? "AND total_pop.cap_code_id = pnad_novo.cod_cap" : ""}
        ${dimensions.metro_code ? "AND pnad_novo.cod_rm_ride <> 99 AND total_pop.metro_code_id = pnad_novo.cod_rm_ride" : ""}
        ${dimensions.region ? "AND total_pop.region_id = pnad_novo.cod_regiao" : ""}
        ${dimensions.state ? "AND total_pop.state_id = pnad_novo.cod_uf" : ""}

    `

    // Subquery: total_apoio_freq
    let totalApoioFreq = squel.select()
        .field("pnad_novo.ano_ref", "ano_ref")
        .field("pnad_novo.apoio_frequencia_ajustada", "apoio_frequencia_ajustada")
        .field("total_pop.total", "total")
        .field("SUM(pnad_novo.peso_domicilio_pessoas_com_cal)", "total_freq")
        .from("pnad_novo")
        .join(
            totalPop, 
            "total_pop",
            joinQuery 
        )
        .group("pnad_novo.ano_ref")
        .group("pnad_novo.apoio_frequencia_ajustada")
        .group("total_pop.total");

    // Aplicar filtros dinâmicos à subquery total_apoio_freq
    totalApoioFreq = rqf.buildQuery(req, totalApoioFreq);

    // Query Principal
    let mainQuery = squel.select()
        .from(totalApoioFreq, "total_apoio_freq")
        .field("total_apoio_freq.ano_ref", "year")
        .field("total_apoio_freq.apoio_frequencia_ajustada", "age_range_all_id")
        .field("ROUND((total_apoio_freq.total_freq / total_apoio_freq.total) * 100, 1)", "total")
        .where("total_apoio_freq.ano_ref >= 2019")
        .order("total_apoio_freq.ano_ref")
        .order("total_apoio_freq.apoio_frequencia_ajustada");

    if (dimensions.gender)
        mainQuery.field("total_apoio_freq.gender_id")

    if (dimensions.bolsa_familia)
        mainQuery.field("total_apoio_freq.bolsa_familia_id")

    if (dimensions.new_pnad_ethnic_group)
        mainQuery.field("total_apoio_freq.new_pnad_ethnic_group_id")

    if (dimensions.income_range)
        mainQuery.field("total_apoio_freq.income_range_id")

    if (dimensions.cap_code)
        mainQuery.field("total_apoio_freq.cap_code_id")

    if (dimensions.metro_code)
        mainQuery.field("total_apoio_freq.metro_code_id")

    if (dimensions.region)
        mainQuery.field("total_apoio_freq.region_id")

    if (dimensions.state)
        mainQuery.field("total_apoio_freq.state_id")

    req.sql = mainQuery;    

    next();
}, query, id2str.transform(false), response('adjusted_liquid_frequency'));

module.exports = adjustedLiquidFrequency;
